import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Skeleton } from "@/components/ui/skeleton";
import { Link } from "wouter";
import {
  FileText,
  Database,
  Clock,
  TrendingUp,
  AlertTriangle,
  Activity,
  BarChart3,
  Upload,
  ArrowRight,
  RefreshCw,
} from "lucide-react";
import {
  AreaChart,
  Area,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  BarChart,
  Bar,
  PieChart,
  Pie,
  Cell,
} from "recharts";

const COLORS = ["hsl(217, 91%, 60%)", "hsl(160, 60%, 45%)", "hsl(30, 80%, 55%)", "hsl(280, 65%, 60%)", "hsl(340, 75%, 55%)"];

interface DashboardStats {
  totalFiles: number;
  totalRecords: number;
  recentUploads: number;
  avgProcessingTime: string;
}

interface RecentFile {
  id: number;
  regName: string;
  secName: string;
  uploadDate: string;
  isParsed: boolean;
}

export default function Dashboard() {
  const { data: stats, isLoading: statsLoading } = useQuery<DashboardStats>({
    queryKey: ["/api/dashboard/stats"],
  });

  const { data: recentFiles, isLoading: filesLoading } = useQuery<RecentFile[]>({
    queryKey: ["/api/files/recent"],
  });

  const { data: chartData } = useQuery<{ date: string; uploads: number; records: number }[]>({
    queryKey: ["/api/dashboard/chart"],
  });

  const { data: tableStats } = useQuery<{ name: string; count: number }[]>({
    queryKey: ["/api/dashboard/tables"],
  });

  const mockChartData = chartData || [
    { date: "Jan", uploads: 12, records: 340 },
    { date: "Feb", uploads: 19, records: 520 },
    { date: "Mar", uploads: 15, records: 410 },
    { date: "Apr", uploads: 25, records: 680 },
    { date: "May", uploads: 22, records: 590 },
    { date: "Jun", uploads: 30, records: 820 },
  ];

  const mockTableStats = tableStats || [
    { name: "Bit Data", count: 245 },
    { name: "Drill String", count: 189 },
    { name: "Mud Data", count: 312 },
    { name: "Companies", count: 156 },
    { name: "Lost Time", count: 78 },
  ];

  return (
    <div className="flex-1 overflow-auto p-6 space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
        <div>
          <h1 className="text-2xl font-bold tracking-tight">Dashboard</h1>
          <p className="text-muted-foreground">Overview of your drilling reports data</p>
        </div>
        <div className="flex items-center gap-2">
          <Button variant="outline" size="sm" data-testid="button-refresh">
            <RefreshCw className="h-4 w-4 mr-2" />
            Refresh
          </Button>
          <Button asChild size="sm" data-testid="button-upload-new">
            <Link href="/upload">
              <Upload className="h-4 w-4 mr-2" />
              Upload Files
            </Link>
          </Button>
        </div>
      </div>

      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        <Card data-testid="card-total-files">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2 gap-2">
            <CardTitle className="text-sm font-medium">Total Files</CardTitle>
            <FileText className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {statsLoading ? (
              <Skeleton className="h-8 w-20" />
            ) : (
              <>
                <div className="text-2xl font-bold" data-testid="text-total-files">
                  {stats?.totalFiles || 0}
                </div>
                <p className="text-xs text-muted-foreground">
                  <span className="text-green-500">+12%</span> from last month
                </p>
              </>
            )}
          </CardContent>
        </Card>

        <Card data-testid="card-total-records">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2 gap-2">
            <CardTitle className="text-sm font-medium">Total Records</CardTitle>
            <Database className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {statsLoading ? (
              <Skeleton className="h-8 w-20" />
            ) : (
              <>
                <div className="text-2xl font-bold" data-testid="text-total-records">
                  {stats?.totalRecords?.toLocaleString() || 0}
                </div>
                <p className="text-xs text-muted-foreground">
                  Across all tables
                </p>
              </>
            )}
          </CardContent>
        </Card>

        <Card data-testid="card-recent-uploads">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2 gap-2">
            <CardTitle className="text-sm font-medium">Recent Uploads</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {statsLoading ? (
              <Skeleton className="h-8 w-20" />
            ) : (
              <>
                <div className="text-2xl font-bold" data-testid="text-recent-uploads">
                  {stats?.recentUploads || 0}
                </div>
                <p className="text-xs text-muted-foreground">
                  In the last 7 days
                </p>
              </>
            )}
          </CardContent>
        </Card>

        <Card data-testid="card-processing-time">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2 gap-2">
            <CardTitle className="text-sm font-medium">Avg Processing</CardTitle>
            <Clock className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            {statsLoading ? (
              <Skeleton className="h-8 w-20" />
            ) : (
              <>
                <div className="text-2xl font-bold" data-testid="text-processing-time">
                  {stats?.avgProcessingTime || "2.3s"}
                </div>
                <p className="text-xs text-muted-foreground">
                  Per file average
                </p>
              </>
            )}
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-4 lg:grid-cols-7">
        <Card className="lg:col-span-4" data-testid="card-activity-chart">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Activity className="h-5 w-5" />
              Upload Activity
            </CardTitle>
            <CardDescription>
              File uploads and records over the past 6 months
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-[300px]">
              <ResponsiveContainer width="100%" height="100%">
                <AreaChart data={mockChartData}>
                  <defs>
                    <linearGradient id="colorUploads" x1="0" y1="0" x2="0" y2="1">
                      <stop offset="5%" stopColor="hsl(217, 91%, 60%)" stopOpacity={0.3} />
                      <stop offset="95%" stopColor="hsl(217, 91%, 60%)" stopOpacity={0} />
                    </linearGradient>
                  </defs>
                  <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                  <XAxis dataKey="date" stroke="hsl(var(--muted-foreground))" fontSize={12} />
                  <YAxis stroke="hsl(var(--muted-foreground))" fontSize={12} />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: "hsl(var(--card))",
                      border: "1px solid hsl(var(--border))",
                      borderRadius: "8px",
                    }}
                  />
                  <Area
                    type="monotone"
                    dataKey="uploads"
                    stroke="hsl(217, 91%, 60%)"
                    fillOpacity={1}
                    fill="url(#colorUploads)"
                    strokeWidth={2}
                  />
                </AreaChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>

        <Card className="lg:col-span-3" data-testid="card-data-distribution">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <BarChart3 className="h-5 w-5" />
              Data Distribution
            </CardTitle>
            <CardDescription>
              Records by table type
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="h-[300px]">
              <ResponsiveContainer width="100%" height="100%">
                <BarChart data={mockTableStats} layout="vertical">
                  <CartesianGrid strokeDasharray="3 3" stroke="hsl(var(--border))" />
                  <XAxis type="number" stroke="hsl(var(--muted-foreground))" fontSize={12} />
                  <YAxis dataKey="name" type="category" stroke="hsl(var(--muted-foreground))" fontSize={12} width={80} />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: "hsl(var(--card))",
                      border: "1px solid hsl(var(--border))",
                      borderRadius: "8px",
                    }}
                  />
                  <Bar dataKey="count" fill="hsl(217, 91%, 60%)" radius={[0, 4, 4, 0]} />
                </BarChart>
              </ResponsiveContainer>
            </div>
          </CardContent>
        </Card>
      </div>

      <div className="grid gap-4 lg:grid-cols-2">
        <Card data-testid="card-recent-files">
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle>Recent Files</CardTitle>
                <CardDescription>Latest uploaded drilling reports</CardDescription>
              </div>
              <Button variant="ghost" size="sm" asChild>
                <Link href="/upload">
                  View All
                  <ArrowRight className="ml-2 h-4 w-4" />
                </Link>
              </Button>
            </div>
          </CardHeader>
          <CardContent>
            {filesLoading ? (
              <div className="space-y-4">
                {[1, 2, 3].map((i) => (
                  <div key={i} className="flex items-center gap-4">
                    <Skeleton className="h-10 w-10 rounded" />
                    <div className="space-y-2 flex-1">
                      <Skeleton className="h-4 w-full" />
                      <Skeleton className="h-3 w-24" />
                    </div>
                  </div>
                ))}
              </div>
            ) : recentFiles && recentFiles.length > 0 ? (
              <div className="space-y-4">
                {recentFiles.slice(0, 5).map((file) => (
                  <div key={file.id} className="flex items-center gap-4" data-testid={`file-item-${file.id}`}>
                    <div className="flex h-10 w-10 items-center justify-center rounded bg-primary/10">
                      <FileText className="h-5 w-5 text-primary" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <p className="text-sm font-medium truncate">{file.regName}</p>
                      <p className="text-xs text-muted-foreground">{file.secName}</p>
                    </div>
                    <div className="text-right">
                      <Badge variant={file.isParsed ? "default" : "secondary"}>
                        {file.isParsed ? "Parsed" : "Pending"}
                      </Badge>
                      <p className="text-xs text-muted-foreground mt-1">{file.uploadDate}</p>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="flex flex-col items-center justify-center py-8 text-center">
                <FileText className="h-12 w-12 text-muted-foreground/50 mb-4" />
                <p className="text-muted-foreground">No files uploaded yet</p>
                <Button variant="link" asChild className="mt-2">
                  <Link href="/upload">Upload your first file</Link>
                </Button>
              </div>
            )}
          </CardContent>
        </Card>

        <Card data-testid="card-quick-actions">
          <CardHeader>
            <CardTitle>Quick Actions</CardTitle>
            <CardDescription>Common tasks and shortcuts</CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid grid-cols-2 gap-4">
              <Button variant="outline" className="h-auto py-4 flex-col gap-2" asChild data-testid="quick-action-upload">
                <Link href="/upload">
                  <Upload className="h-6 w-6" />
                  <span className="text-sm">Upload Files</span>
                </Link>
              </Button>
              <Button variant="outline" className="h-auto py-4 flex-col gap-2" asChild data-testid="quick-action-query">
                <Link href="/query-builder">
                  <Database className="h-6 w-6" />
                  <span className="text-sm">Query Builder</span>
                </Link>
              </Button>
              <Button variant="outline" className="h-auto py-4 flex-col gap-2" asChild data-testid="quick-action-chat">
                <Link href="/chat">
                  <Activity className="h-6 w-6" />
                  <span className="text-sm">AI Assistant</span>
                </Link>
              </Button>
              <Button variant="outline" className="h-auto py-4 flex-col gap-2" asChild data-testid="quick-action-reports">
                <Link href="/reports">
                  <BarChart3 className="h-6 w-6" />
                  <span className="text-sm">View Reports</span>
                </Link>
              </Button>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
