import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Collapsible,
  CollapsibleContent,
  CollapsibleTrigger,
} from "@/components/ui/collapsible";
import { FileText, Play, Trash2, CheckCircle, Clock, AlertCircle, ChevronDown, ChevronRight, Calendar, Folder, FolderOpen } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";

interface FileRecord {
  id: number;
  regName: string;
  secName: string;
  filePath: string;
  uploadDate: string;
  isParsed: boolean;
  createdAt: string;
}

interface GroupedFiles {
  [year: string]: {
    [month: string]: FileRecord[];
  };
}

const MONTH_NAMES = [
  "January", "February", "March", "April", "May", "June",
  "July", "August", "September", "October", "November", "December"
];

export default function FilesTablePage() {
  const { toast } = useToast();
  const [openYears, setOpenYears] = useState<Set<string>>(new Set());
  const [openMonths, setOpenMonths] = useState<Set<string>>(new Set());

  const { data: files, isLoading } = useQuery<FileRecord[]>({
    queryKey: ["/api/files"],
  });

  const parseMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await apiRequest("POST", `/api/files/${id}/parse`);
      return response.json();
    },
    onSuccess: (data: any) => {
      queryClient.invalidateQueries({ queryKey: ["/api/files"] });
      queryClient.invalidateQueries({ queryKey: ["/api/rigs"] });
      toast({
        title: "PDF Parsed Successfully",
        description: `Extracted ${data.reportCount} reports. Rigs found: ${data.rigNames?.join(", ") || "None"}`,
      });
    },
    onError: () => {
      toast({
        title: "Parse Failed",
        description: "Failed to parse PDF file",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/files/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/files"] });
      toast({
        title: "File Deleted",
        description: "File has been removed from the system",
      });
    },
    onError: () => {
      toast({
        title: "Delete Failed",
        description: "Failed to delete file",
        variant: "destructive",
      });
    },
  });

  const groupFilesByYearMonth = (files: FileRecord[]): GroupedFiles => {
    const grouped: GroupedFiles = {};
    
    files.forEach(file => {
      const date = new Date(file.uploadDate);
      const year = date.getFullYear().toString();
      const month = date.getMonth().toString();
      
      if (!grouped[year]) {
        grouped[year] = {};
      }
      if (!grouped[year][month]) {
        grouped[year][month] = [];
      }
      grouped[year][month].push(file);
    });
    
    return grouped;
  };

  const formatDate = (dateStr: string) => {
    return new Date(dateStr).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  const getFileName = (filePath: string) => {
    const parts = filePath.split("-");
    return parts.length > 2 ? parts.slice(2).join("-") : filePath;
  };

  const toggleYear = (year: string) => {
    setOpenYears(prev => {
      const next = new Set(prev);
      if (next.has(year)) {
        next.delete(year);
      } else {
        next.add(year);
      }
      return next;
    });
  };

  const toggleMonth = (yearMonth: string) => {
    setOpenMonths(prev => {
      const next = new Set(prev);
      if (next.has(yearMonth)) {
        next.delete(yearMonth);
      } else {
        next.add(yearMonth);
      }
      return next;
    });
  };

  const groupedFiles = files ? groupFilesByYearMonth(files) : {};
  const sortedYears = Object.keys(groupedFiles).sort((a, b) => parseInt(b) - parseInt(a));

  const getTotalFilesForYear = (yearData: { [month: string]: FileRecord[] }) => {
    return Object.values(yearData).reduce((sum, files) => sum + files.length, 0);
  };

  return (
    <div className="p-6 h-full overflow-hidden flex flex-col">
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Files Table</h1>
          <p className="text-muted-foreground">View and manage all uploaded PDF files organized by date</p>
        </div>
        <Badge variant="outline" className="text-sm">
          {files?.length || 0} Files
        </Badge>
      </div>

      <Card className="flex-1 overflow-hidden flex flex-col">
        <CardHeader className="flex-shrink-0 pb-3 border-b">
          <CardTitle className="text-lg flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Files by Year / Month
          </CardTitle>
        </CardHeader>
        <CardContent className="flex-1 overflow-hidden p-0">
          {isLoading ? (
            <div className="p-4 space-y-3">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : sortedYears.length > 0 ? (
            <ScrollArea className="h-[calc(100vh-280px)]">
              <div className="p-4 space-y-2">
                {sortedYears.map(year => (
                  <Collapsible
                    key={year}
                    open={openYears.has(year)}
                    onOpenChange={() => toggleYear(year)}
                  >
                    <CollapsibleTrigger asChild>
                      <div 
                        className="flex items-center gap-3 p-3 rounded-lg bg-muted/50 hover-elevate cursor-pointer"
                        data-testid={`folder-year-${year}`}
                      >
                        {openYears.has(year) ? (
                          <ChevronDown className="h-5 w-5 text-muted-foreground" />
                        ) : (
                          <ChevronRight className="h-5 w-5 text-muted-foreground" />
                        )}
                        {openYears.has(year) ? (
                          <FolderOpen className="h-5 w-5 text-primary" />
                        ) : (
                          <Folder className="h-5 w-5 text-primary" />
                        )}
                        <span className="font-semibold text-lg">{year}</span>
                        <Badge variant="secondary" className="ml-auto">
                          {getTotalFilesForYear(groupedFiles[year])} Files
                        </Badge>
                      </div>
                    </CollapsibleTrigger>
                    <CollapsibleContent>
                      <div className="ml-6 mt-2 space-y-2">
                        {Object.keys(groupedFiles[year])
                          .sort((a, b) => parseInt(b) - parseInt(a))
                          .map(month => {
                            const yearMonth = `${year}-${month}`;
                            const monthFiles = groupedFiles[year][month];
                            return (
                              <Collapsible
                                key={yearMonth}
                                open={openMonths.has(yearMonth)}
                                onOpenChange={() => toggleMonth(yearMonth)}
                              >
                                <CollapsibleTrigger asChild>
                                  <div 
                                    className="flex items-center gap-3 p-2 rounded-lg bg-background border hover-elevate cursor-pointer"
                                    data-testid={`folder-month-${yearMonth}`}
                                  >
                                    {openMonths.has(yearMonth) ? (
                                      <ChevronDown className="h-4 w-4 text-muted-foreground" />
                                    ) : (
                                      <ChevronRight className="h-4 w-4 text-muted-foreground" />
                                    )}
                                    {openMonths.has(yearMonth) ? (
                                      <FolderOpen className="h-4 w-4 text-blue-500" />
                                    ) : (
                                      <Folder className="h-4 w-4 text-blue-500" />
                                    )}
                                    <span className="font-medium">{MONTH_NAMES[parseInt(month)]}</span>
                                    <Badge variant="outline" className="ml-auto">
                                      {monthFiles.length} Files
                                    </Badge>
                                  </div>
                                </CollapsibleTrigger>
                                <CollapsibleContent>
                                  <div className="ml-6 mt-2 space-y-1">
                                    {monthFiles.map(file => (
                                      <div
                                        key={file.id}
                                        className="flex items-center gap-3 p-3 rounded-lg bg-card border hover-elevate"
                                        data-testid={`row-file-${file.id}`}
                                      >
                                        <FileText className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                                        <div className="flex-1 min-w-0">
                                          <div className="font-medium truncate">
                                            {getFileName(file.filePath)}
                                          </div>
                                          <div className="flex items-center gap-2 text-sm text-muted-foreground">
                                            <span>{formatDate(file.uploadDate)}</span>
                                            <span>•</span>
                                            <span>{file.secName || "Daily Report"}</span>
                                          </div>
                                        </div>
                                        <div className="flex items-center gap-2 flex-shrink-0">
                                          {file.isParsed ? (
                                            <Badge variant="default" className="bg-green-600">
                                              <CheckCircle className="h-3 w-3 mr-1" />
                                              Parsed
                                            </Badge>
                                          ) : (
                                            <Badge variant="outline">
                                              <Clock className="h-3 w-3 mr-1" />
                                              Pending
                                            </Badge>
                                          )}
                                          {!file.isParsed && (
                                            <Button
                                              size="sm"
                                              variant="outline"
                                              onClick={(e) => {
                                                e.stopPropagation();
                                                parseMutation.mutate(file.id);
                                              }}
                                              disabled={parseMutation.isPending}
                                              data-testid={`button-parse-${file.id}`}
                                            >
                                              <Play className="h-3 w-3 mr-1" />
                                              Parse
                                            </Button>
                                          )}
                                          <Button
                                            size="sm"
                                            variant="ghost"
                                            onClick={(e) => {
                                              e.stopPropagation();
                                              deleteMutation.mutate(file.id);
                                            }}
                                            disabled={deleteMutation.isPending}
                                            data-testid={`button-delete-${file.id}`}
                                          >
                                            <Trash2 className="h-3 w-3 text-destructive" />
                                          </Button>
                                        </div>
                                      </div>
                                    ))}
                                  </div>
                                </CollapsibleContent>
                              </Collapsible>
                            );
                          })}
                      </div>
                    </CollapsibleContent>
                  </Collapsible>
                ))}
              </div>
            </ScrollArea>
          ) : (
            <div className="flex flex-col items-center justify-center h-full py-12 text-muted-foreground">
              <AlertCircle className="h-12 w-12 mb-4 opacity-50" />
              <p className="text-lg font-medium">No files uploaded yet</p>
              <p className="text-sm">Upload PDF files to see them organized by year and month</p>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
