import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { ThemeToggle } from "@/components/theme-toggle";
import {
  Database,
  FileSearch,
  MessageSquare,
  BarChart3,
  Shield,
  Zap,
  ArrowRight,
  CheckCircle,
} from "lucide-react";

const features = [
  {
    icon: FileSearch,
    title: "PDF Processing",
    description: "Automatically parse drilling reports and extract structured data from PDF files",
  },
  {
    icon: Database,
    title: "Query Builder",
    description: "Build custom queries to explore your drilling data with an intuitive visual interface",
  },
  {
    icon: MessageSquare,
    title: "AI Assistant",
    description: "Chat with your data using natural language powered by advanced AI",
  },
  {
    icon: BarChart3,
    title: "Analytics Dashboard",
    description: "Visualize KPIs and trends with interactive charts and reports",
  },
  {
    icon: Shield,
    title: "Secure Access",
    description: "Enterprise-grade security with role-based access control",
  },
  {
    icon: Zap,
    title: "Fast Export",
    description: "Export your query results to CSV with a single click",
  },
];

const benefits = [
  "Process hundreds of drilling reports in minutes",
  "No more manual data entry",
  "Ask questions in plain English",
  "Real-time dashboard updates",
];

export default function Landing() {
  return (
    <div className="min-h-screen bg-background">
      <header className="sticky top-0 z-50 w-full border-b border-border bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60">
        <div className="container flex h-16 items-center justify-between px-4 mx-auto max-w-7xl">
          <div className="flex items-center gap-3">
            <div className="flex h-9 w-9 items-center justify-center rounded-md bg-primary text-primary-foreground font-bold text-lg">
              A
            </div>
            <span className="font-semibold text-xl">ARMS</span>
          </div>
          
          <nav className="hidden md:flex items-center gap-6">
            <a href="#features" className="text-muted-foreground hover:text-foreground transition-colors">
              Features
            </a>
            <a href="#benefits" className="text-muted-foreground hover:text-foreground transition-colors">
              Benefits
            </a>
          </nav>
          
          <div className="flex items-center gap-2">
            <ThemeToggle />
            <Button asChild data-testid="button-login">
              <a href="/api/login">
                Sign In
                <ArrowRight className="ml-2 h-4 w-4" />
              </a>
            </Button>
          </div>
        </div>
      </header>

      <main>
        <section className="py-20 md:py-32">
          <div className="container px-4 mx-auto max-w-7xl">
            <div className="flex flex-col items-center text-center max-w-3xl mx-auto">
              <div className="inline-flex items-center rounded-full border border-border px-4 py-1.5 text-sm text-muted-foreground mb-6">
                <Zap className="h-4 w-4 mr-2 text-primary" />
                Advanced Reporting Management System
              </div>
              
              <h1 className="text-4xl md:text-6xl font-bold tracking-tight mb-6">
                Transform Your{" "}
                <span className="text-primary">Drilling Reports</span>{" "}
                Into Insights
              </h1>
              
              <p className="text-lg md:text-xl text-muted-foreground mb-8 max-w-2xl">
                ARMS processes PDF drilling reports into structured data, enabling powerful 
                queries and AI-powered analysis. Stop manual data entry and start making 
                data-driven decisions.
              </p>
              
              <div className="flex flex-col sm:flex-row gap-4">
                <Button size="lg" asChild data-testid="button-get-started">
                  <a href="/api/login">
                    Get Started Free
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </a>
                </Button>
                <Button size="lg" variant="outline" asChild>
                  <a href="#features">Learn More</a>
                </Button>
              </div>
              
              <div className="flex flex-wrap items-center justify-center gap-6 mt-8 text-sm text-muted-foreground">
                <span className="flex items-center gap-2">
                  <CheckCircle className="h-4 w-4 text-green-500" />
                  Free to start
                </span>
                <span className="flex items-center gap-2">
                  <CheckCircle className="h-4 w-4 text-green-500" />
                  No credit card required
                </span>
                <span className="flex items-center gap-2">
                  <CheckCircle className="h-4 w-4 text-green-500" />
                  Sign in with Google
                </span>
              </div>
            </div>
          </div>
        </section>

        <section id="features" className="py-20 bg-muted/30">
          <div className="container px-4 mx-auto max-w-7xl">
            <div className="text-center mb-12">
              <h2 className="text-3xl md:text-4xl font-bold mb-4">
                Everything You Need
              </h2>
              <p className="text-lg text-muted-foreground max-w-2xl mx-auto">
                Powerful tools to process, analyze, and extract insights from your drilling reports
              </p>
            </div>
            
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
              {features.map((feature) => (
                <Card key={feature.title} className="border-border hover-elevate transition-all duration-200">
                  <CardContent className="p-6">
                    <div className="flex h-12 w-12 items-center justify-center rounded-lg bg-primary/10 mb-4">
                      <feature.icon className="h-6 w-6 text-primary" />
                    </div>
                    <h3 className="text-lg font-semibold mb-2">{feature.title}</h3>
                    <p className="text-muted-foreground">{feature.description}</p>
                  </CardContent>
                </Card>
              ))}
            </div>
          </div>
        </section>

        <section id="benefits" className="py-20">
          <div className="container px-4 mx-auto max-w-7xl">
            <div className="grid lg:grid-cols-2 gap-12 items-center">
              <div>
                <h2 className="text-3xl md:text-4xl font-bold mb-6">
                  Save Hours Every Week
                </h2>
                <p className="text-lg text-muted-foreground mb-8">
                  ARMS automates the tedious process of extracting data from drilling reports. 
                  Upload your PDFs and let our system do the heavy lifting.
                </p>
                
                <ul className="space-y-4">
                  {benefits.map((benefit) => (
                    <li key={benefit} className="flex items-center gap-3">
                      <div className="flex h-6 w-6 items-center justify-center rounded-full bg-green-500/10">
                        <CheckCircle className="h-4 w-4 text-green-500" />
                      </div>
                      <span>{benefit}</span>
                    </li>
                  ))}
                </ul>
                
                <Button className="mt-8" size="lg" asChild>
                  <a href="/api/login">
                    Start Free Trial
                    <ArrowRight className="ml-2 h-4 w-4" />
                  </a>
                </Button>
              </div>
              
              <div className="relative">
                <div className="aspect-video rounded-lg bg-gradient-to-br from-primary/20 to-primary/5 border border-border flex items-center justify-center">
                  <div className="text-center p-8">
                    <BarChart3 className="h-16 w-16 text-primary mx-auto mb-4" />
                    <p className="text-muted-foreground">Interactive Dashboard Preview</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </section>
      </main>

      <footer className="border-t border-border py-8">
        <div className="container px-4 mx-auto max-w-7xl">
          <div className="flex flex-col md:flex-row items-center justify-between gap-4">
            <div className="flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-md bg-primary text-primary-foreground font-bold">
                A
              </div>
              <span className="font-semibold">ARMS</span>
            </div>
            <p className="text-sm text-muted-foreground">
              Advanced Reporting Management System. All rights reserved.
            </p>
          </div>
        </div>
      </footer>
    </div>
  );
}
