import { useState, useCallback, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from "@/components/ui/collapsible";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  ChevronRight,
  ChevronDown,
  Play,
  Save,
  Trash2,
  Download,
  Database,
  Filter,
  Calendar,
  Loader2,
  Plus,
  Search,
  X,
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

interface TableConfig {
  name: string;
  displayName: string;
  columns: { name: string; displayName: string }[];
}

const tableConfigurations: TableConfig[] = [
  {
    name: "first_table",
    displayName: "Rig Details",
    columns: [
      { name: "date", displayName: "Date" },
      { name: "well_no_and_charge", displayName: "Well_No_and_Charge" },
      { name: "wellbores", displayName: "Wellbores" },
      { name: "objective", displayName: "Objective" },
      { name: "rig", displayName: "Rig" },
      { name: "foreman_engineer", displayName: "Foreman_Engineer" },
      { name: "engineer", displayName: "Engineer" },
      { name: "manager", displayName: "Manager" },
      { name: "thuraya_rig_forman", displayName: "Thuraya_Rig_Forman" },
      { name: "rig_forman_vsat", displayName: "Rig_Forman_VSAT" },
      { name: "contractor_clerk", displayName: "Contractor_Clerk" },
    ],
  },
  {
    name: "second_table",
    displayName: "Current Operation",
    columns: [
      { name: "depth", displayName: "Depth" },
      { name: "last_24_hr_operations", displayName: "Last_24_hr_operations" },
      { name: "next_24_hr_plan", displayName: "Next_24_hr_plan" },
      { name: "location", displayName: "Location" },
      { name: "next_location", displayName: "next_location" },
      { name: "days_since_spud_and_date", displayName: "Days_Since_Spud_and_Date" },
      { name: "formation_tops", displayName: "Formation_tops" },
      { name: "prev_depth", displayName: "Prev_Depth" },
      { name: "depth_comments", displayName: "Depth_Comments" },
      { name: "liner_size_tol_md_tvd", displayName: "Liner_Size_TOL_MD_TVD" },
      { name: "last_csg_size_md_tvd", displayName: "Last_Csg_Size_MD_TVD" },
      { name: "footage", displayName: "Footage" },
      { name: "distance_from_dha", displayName: "Distance_from_DHA" },
      { name: "circ", displayName: "Circ" },
    ],
  },
  {
    name: "third_table",
    displayName: "Detailed Operation",
    columns: [
      { name: "hrs", displayName: "Hrs" },
      { name: "from_to", displayName: "From_To" },
      { name: "lateral", displayName: "Lateral" },
      { name: "hole_size", displayName: "Hole_size" },
      { name: "cat", displayName: "Cat" },
      { name: "major_op", displayName: "Major_OP" },
      { name: "action", displayName: "Action" },
      { name: "object", displayName: "Object" },
      { name: "resp_co", displayName: "Resp_Co" },
      { name: "hole_depth", displayName: "Hole_depth" },
      { name: "event_depth", displayName: "Event_depth" },
      { name: "summary_of_operations", displayName: "Summary_of_Operations" },
      { name: "lt_type", displayName: "LT_TYPE" },
      { name: "lt_id", displayName: "LT_ID" },
    ],
  },
  {
    name: "lost_time_details",
    displayName: "Lost Time Table",
    columns: [
      { name: "from_to", displayName: "From_To" },
      { name: "hrs", displayName: "Hrs" },
      { name: "cum_hrs", displayName: "Cum_Hrs" },
      { name: "lt_id", displayName: "LT_ID" },
      { name: "parent_lt_id", displayName: "Parent_LT_ID" },
      { name: "type", displayName: "Type" },
      { name: "cause", displayName: "Cause" },
      { name: "object", displayName: "Object" },
      { name: "resp_co", displayName: "Resp_Co" },
      { name: "depth", displayName: "Depth" },
      { name: "lt_summary", displayName: "LT_Summary" },
    ],
  },
  {
    name: "other_data_table",
    displayName: "Comments Table",
    columns: [
      { name: "comment", displayName: "comment" },
    ],
  },
  {
    name: "fourth_table",
    displayName: "Lateral & TVD",
    columns: [
      { name: "lateral", displayName: "Lateral" },
      { name: "survey_md", displayName: "Survey_MD" },
      { name: "angle", displayName: "Angle" },
      { name: "azimuth", displayName: "Azimuth" },
      { name: "c_length", displayName: "CLength" },
      { name: "tvd", displayName: "TVD" },
      { name: "ns_coordinate", displayName: "NS_Coordinate" },
      { name: "ew_coordinate", displayName: "EW_Coordinate" },
      { name: "vertical_sec", displayName: "Vertical_Sec" },
      { name: "dls", displayName: "DLS" },
    ],
  },
  {
    name: "mud_data",
    displayName: "MUD Data Table",
    columns: [
      { name: "weight_pcf", displayName: "Weight_PCF" },
      { name: "funnel_sec", displayName: "Funnel_SEC" },
      { name: "filtrate_wl_hthp", displayName: "FiltrateWL_HTHP" },
      { name: "filtrate_wl_api", displayName: "FiltrateWL_API" },
      { name: "pv", displayName: "PV" },
      { name: "yp", displayName: "YP" },
      { name: "electrical_stability_volts", displayName: "Electrical_Stability_Volts" },
      { name: "rpm_3_6", displayName: "RPM_3_6" },
      { name: "gels_sec_min", displayName: "Gels_Sec_Min" },
      { name: "ph", displayName: "PH" },
      { name: "ca_ppm", displayName: "CA_PPM" },
      { name: "cl_ppm", displayName: "CL_PPM" },
      { name: "fl_temp", displayName: "FL_Temp" },
      { name: "cake_hthp", displayName: "Cake_HTHP" },
      { name: "cake_api", displayName: "Cake_API" },
      { name: "water_vol", displayName: "Water_Vol" },
      { name: "oil_vol", displayName: "Oil_Vol" },
      { name: "solids_vol", displayName: "Solids_Vol" },
      { name: "sand_vol", displayName: "Sand_Vol" },
      { name: "lgs", displayName: "LGS" },
      { name: "mbt", displayName: "MBT" },
      { name: "mud_type", displayName: "Mud_type" },
      { name: "ppt_spurt", displayName: "PPT_Spurt" },
      { name: "ppt_total", displayName: "PPT_Total" },
      { name: "water_well", displayName: "Water_Well" },
    ],
  },
  {
    name: "mud_treatment",
    displayName: "MUD Treatment Table",
    columns: [
      { name: "material", displayName: "material" },
      { name: "quantity", displayName: "quantity" },
      { name: "unit", displayName: "unit" },
      { name: "mud_treatment", displayName: "MUD_TREATMENT" },
    ],
  },
  {
    name: "bit_data",
    displayName: "Bit Data",
    columns: [
      { name: "bit_number", displayName: "Bit_Number" },
      { name: "iadc", displayName: "IADC" },
      { name: "size", displayName: "Size" },
      { name: "manufacturer", displayName: "Manufacturer" },
      { name: "serial_number", displayName: "Serial_Number" },
      { name: "type", displayName: "Type" },
      { name: "jets", displayName: "Jets" },
      { name: "tfa", displayName: "TFA" },
      { name: "md_in", displayName: "MD_In" },
      { name: "md_out", displayName: "MD_Out" },
      { name: "tvd_out", displayName: "TVD_Out" },
      { name: "feet", displayName: "Feet" },
      { name: "hours", displayName: "Hours" },
      { name: "average_rop", displayName: "Average_ROP" },
      { name: "wob", displayName: "WOB" },
      { name: "rpm", displayName: "RPM" },
      { name: "i_row", displayName: "I_Row" },
      { name: "o_row", displayName: "O_Row" },
      { name: "dc", displayName: "DC" },
      { name: "loc", displayName: "Loc" },
      { name: "b", displayName: "B" },
      { name: "g", displayName: "G" },
      { name: "pressure", displayName: "Pressure" },
      { name: "gpm", displayName: "GPM" },
      { name: "jet_vel", displayName: "Jet_Vel" },
      { name: "dp_av", displayName: "DP_Av" },
      { name: "dc_av", displayName: "DC_Av" },
      { name: "bit_hhp", displayName: "Bit_HHP" },
      { name: "other", displayName: "Other" },
      { name: "pooh_reason", displayName: "POOH_Reason" },
    ],
  },
  {
    name: "drill_string_floats",
    displayName: "Drill Strings Floats",
    columns: [
      { name: "float", displayName: "Float" },
      { name: "bha_hrs", displayName: "BHA_Hrs" },
      { name: "string_wt", displayName: "String_Wt" },
      { name: "pick_up", displayName: "Pick_Up" },
      { name: "slack_off", displayName: "Slack_Off" },
      { name: "rot_torque", displayName: "Rot_Torque" },
      { name: "jars", displayName: "Jars" },
      { name: "serial_hrs", displayName: "Serial_Hrs" },
      { name: "shock", displayName: "Shock" },
      { name: "sub", displayName: "Sub" },
      { name: "mud_motor", displayName: "Mud_Motor" },
    ],
  },
  {
    name: "drill_string",
    displayName: "Drill Strings",
    columns: [
      { name: "order_num", displayName: "OrderNum" },
      { name: "component", displayName: "Component" },
      { name: "provider", displayName: "Provider" },
      { name: "nominal_size", displayName: "Nominal_Size" },
      { name: "joints", displayName: "Joints" },
      { name: "od_size", displayName: "OD_Size" },
      { name: "id_size", displayName: "ID_Size" },
      { name: "length", displayName: "Length" },
      { name: "top_thread", displayName: "Top_Thread" },
      { name: "bottom_thread", displayName: "Bottom_Thread" },
      { name: "weight", displayName: "Weight" },
      { name: "grade", displayName: "Grade" },
      { name: "class", displayName: "Class" },
      { name: "serial", displayName: "Serial" },
    ],
  },
  {
    name: "miscellaneous",
    displayName: "Miscellaneous Table",
    columns: [
      { name: "bop_test", displayName: "BOP_Test" },
      { name: "bop_drills", displayName: "BOP_Drills" },
      { name: "wind", displayName: "Wind" },
      { name: "sea", displayName: "Sea" },
      { name: "weather", displayName: "Weather" },
      { name: "dslta", displayName: "DSLTA" },
      { name: "safety_meeting", displayName: "Safety_Meeting" },
    ],
  },
  {
    name: "companies",
    displayName: "Services Information Table",
    columns: [
      { name: "company", displayName: "Company" },
      { name: "category", displayName: "Category" },
      { name: "no_of_persons", displayName: "No_of_Persons" },
      { name: "operating_hrs", displayName: "Operating_Hrs" },
      { name: "on_location_hrs", displayName: "On_Location_Hrs" },
      { name: "total", displayName: "total" },
    ],
  },
  {
    name: "trucks",
    displayName: "Trucks/Boats Table",
    columns: [
      { name: "truck", displayName: "truck" },
    ],
  },
  {
    name: "bulks",
    displayName: "Bulks Table",
    columns: [
      { name: "bulk", displayName: "BULK" },
      { name: "value", displayName: "Value" },
      { name: "drill_wtr", displayName: "Drill_Wtr" },
      { name: "pot_wtr", displayName: "Pot_Wtr" },
      { name: "fuel_bbls", displayName: "Fuel_BBLS" },
      { name: "barite_sx", displayName: "Barite_SX" },
      { name: "bentonite_sx", displayName: "Bentonite_SX" },
      { name: "cement_g_sx", displayName: "Cement_G_SX" },
    ],
  },
  {
    name: "repair_table",
    displayName: "Repair Table",
    columns: [
      { name: "instrumentation", displayName: "Instrumentation" },
      { name: "other", displayName: "Other" },
      { name: "reporter", displayName: "Reporter" },
      { name: "computer", displayName: "Computer" },
    ],
  },
];

interface QueryResult {
  columns: string[];
  rows: Record<string, any>[];
  totalRows: number;
}

export default function QueryBuilder() {
  const { toast } = useToast();
  const [expandedTables, setExpandedTables] = useState<Set<string>>(new Set());
  const [selectedColumns, setSelectedColumns] = useState<Map<string, Set<string>>>(new Map());
  const [selectedRigs, setSelectedRigs] = useState<Set<string>>(new Set());
  const [rigDialogOpen, setRigDialogOpen] = useState(false);
  const [rigSearchQuery, setRigSearchQuery] = useState("");
  const [dateFrom, setDateFrom] = useState("");
  const [dateTo, setDateTo] = useState("");
  const [queryResult, setQueryResult] = useState<QueryResult | null>(null);

  const { data: availableRigs = [], isLoading: rigsLoading } = useQuery<string[]>({
    queryKey: ["/api/rigs"],
  });

  const rigName = useMemo(() => Array.from(selectedRigs).join(", "), [selectedRigs]);

  const filteredRigs = useMemo(() => {
    if (!rigSearchQuery) return availableRigs;
    return availableRigs.filter(rig => 
      rig.toLowerCase().includes(rigSearchQuery.toLowerCase())
    );
  }, [availableRigs, rigSearchQuery]);

  const toggleRigSelection = (rig: string) => {
    setSelectedRigs(prev => {
      const next = new Set(prev);
      if (next.has(rig)) {
        next.delete(rig);
      } else {
        next.add(rig);
      }
      return next;
    });
  };

  const removeRig = (rig: string) => {
    setSelectedRigs(prev => {
      const next = new Set(prev);
      next.delete(rig);
      return next;
    });
  };

  const toggleTable = (tableName: string) => {
    setExpandedTables((prev) => {
      if (prev.has(tableName)) {
        return new Set();
      } else {
        return new Set([tableName]);
      }
    });
  };

  const toggleColumn = (tableName: string, columnName: string) => {
    setSelectedColumns((prev) => {
      const next = new Map(prev);
      const tableColumns = next.get(tableName) || new Set();
      const newTableColumns = new Set(tableColumns);
      
      if (newTableColumns.has(columnName)) {
        newTableColumns.delete(columnName);
      } else {
        newTableColumns.add(columnName);
      }
      
      if (newTableColumns.size === 0) {
        next.delete(tableName);
      } else {
        next.set(tableName, newTableColumns);
      }
      
      return next;
    });
  };

  const selectAllColumns = (tableName: string, columns: { name: string }[]) => {
    setSelectedColumns((prev) => {
      const next = new Map(prev);
      const allColumns = new Set(columns.map((c) => c.name));
      next.set(tableName, allColumns);
      return next;
    });
  };

  const clearQuery = () => {
    setSelectedColumns(new Map());
    setSelectedRigs(new Set());
    setDateFrom("");
    setDateTo("");
    setQueryResult(null);
  };

  const executeQueryMutation = useMutation({
    mutationFn: async () => {
      const tables: Record<string, string[]> = {};
      selectedColumns.forEach((columns, table) => {
        tables[table] = Array.from(columns);
      });

      const response = await apiRequest("POST", "/api/query/execute", {
        tables,
        rigName: rigName || undefined,
        dateFrom: dateFrom || undefined,
        dateTo: dateTo || undefined,
      });
      return response.json();
    },
    onSuccess: (data) => {
      setQueryResult(data);
      toast({
        title: "Query executed",
        description: `Found ${data.totalRows} records`,
      });
    },
    onError: (error) => {
      toast({
        title: "Query failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const saveQueryMutation = useMutation({
    mutationFn: async () => {
      const tables: Record<string, string[]> = {};
      selectedColumns.forEach((columns, table) => {
        tables[table] = Array.from(columns);
      });

      const response = await apiRequest("POST", "/api/query/save", {
        name: `Query ${new Date().toLocaleString()}`,
        config: {
          tables,
          rigName,
          dateFrom,
          dateTo,
        },
      });
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "Query saved",
        description: "Your query has been saved successfully",
      });
    },
    onError: (error) => {
      toast({
        title: "Save failed",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const exportToCsv = async () => {
    if (!queryResult || queryResult.rows.length === 0) return;

    try {
      const tables: Record<string, string[]> = {};
      selectedColumns.forEach((cols, table) => {
        tables[table] = Array.from(cols);
      });

      const response = await fetch("/api/query/export", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({ tables, rigName, dateFrom, dateTo }),
      });

      if (!response.ok) {
        throw new Error("Export failed");
      }

      const blob = await response.blob();
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = `arms-export-${new Date().toISOString().split("T")[0]}.csv`;
      a.click();
      URL.revokeObjectURL(url);

      toast({
        title: "Export complete",
        description: "Your data has been exported to CSV",
      });
    } catch (error) {
      toast({
        title: "Export failed",
        description: "Failed to export data to CSV",
        variant: "destructive",
      });
    }
  };

  const selectedTablesCount = selectedColumns.size;
  const selectedColumnsCount = Array.from(selectedColumns.values()).reduce(
    (acc, cols) => acc + cols.size,
    0
  );

  return (
    <div className="flex-1 overflow-hidden flex flex-col">
      <div className="flex-none p-6 pb-0">
        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4 mb-6">
          <div>
            <h1 className="text-2xl font-bold tracking-tight">Query Builder</h1>
            <p className="text-muted-foreground">
              Select tables and columns to query your drilling data
            </p>
          </div>
          {selectedColumnsCount > 0 && (
            <div className="flex items-center gap-2">
              <Badge variant="secondary">
                {selectedTablesCount} tables, {selectedColumnsCount} columns selected
              </Badge>
            </div>
          )}
        </div>
      </div>

      <div className="flex-1 overflow-hidden grid lg:grid-cols-12 gap-4 p-6 pt-0">
        <Card className="lg:col-span-3 flex flex-col max-h-[calc(100vh-200px)]" data-testid="card-tables">
          <CardHeader className="flex-none py-4">
            <CardTitle className="text-base flex items-center gap-2">
              <Database className="h-4 w-4" />
              Tables
            </CardTitle>
          </CardHeader>
          <CardContent className="flex-1 overflow-hidden p-0">
            <ScrollArea className="h-[calc(100vh-280px)] px-4 pb-4">
              <div className="space-y-1">
                {tableConfigurations.map((table, index) => (
                  <Collapsible
                    key={table.name}
                    open={expandedTables.has(table.name)}
                    onOpenChange={() => toggleTable(table.name)}
                  >
                    <CollapsibleTrigger className="flex items-center gap-2 w-full p-2 rounded-md hover-elevate text-left text-sm" data-testid={`table-trigger-${table.name}`}>
                      {expandedTables.has(table.name) ? (
                        <ChevronDown className="h-4 w-4 text-muted-foreground" />
                      ) : (
                        <ChevronRight className="h-4 w-4 text-muted-foreground" />
                      )}
                      <span className="text-muted-foreground mr-1">{index + 1}.</span>
                      <span className="flex-1">- {table.displayName}</span>
                      {selectedColumns.has(table.name) && (
                        <Badge variant="secondary" className="text-xs">
                          {selectedColumns.get(table.name)?.size}
                        </Badge>
                      )}
                    </CollapsibleTrigger>
                    <CollapsibleContent>
                      <div className="ml-6 pl-2 border-l border-border space-y-1 py-2">
                        {table.columns.map((column) => (
                          <label
                            key={column.name}
                            className="flex items-center gap-2 p-1.5 rounded hover-elevate cursor-pointer text-sm"
                            data-testid={`column-${table.name}-${column.name}`}
                          >
                            <Checkbox
                              checked={selectedColumns.get(table.name)?.has(column.name) || false}
                              onCheckedChange={() => toggleColumn(table.name, column.name)}
                            />
                            <span className="text-muted-foreground">{column.displayName}</span>
                          </label>
                        ))}
                      </div>
                    </CollapsibleContent>
                  </Collapsible>
                ))}
              </div>
            </ScrollArea>
          </CardContent>
        </Card>

        <div className="lg:col-span-6 flex flex-col gap-4 overflow-hidden">
          <Card className="flex-none" data-testid="card-conditions">
            <CardHeader className="py-4">
              <CardTitle className="text-base flex items-center gap-2">
                <Filter className="h-4 w-4" />
                Conditions
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid gap-4 sm:grid-cols-2">
                <div className="space-y-2">
                  <Label>Rig Name</Label>
                  <div className="flex gap-2">
                    <div className="flex-1 min-h-[36px] flex flex-wrap gap-1 p-2 border rounded-md bg-background">
                      {selectedRigs.size === 0 ? (
                        <span className="text-sm text-muted-foreground">Select rigs...</span>
                      ) : (
                        Array.from(selectedRigs).map(rig => (
                          <Badge key={rig} variant="secondary" className="gap-1" data-testid={`badge-rig-${rig}`}>
                            {rig}
                            <button
                              type="button"
                              onClick={() => removeRig(rig)}
                              className="ml-1 hover:text-destructive"
                            >
                              <X className="h-3 w-3" />
                            </button>
                          </Badge>
                        ))
                      )}
                    </div>
                    <Dialog open={rigDialogOpen} onOpenChange={setRigDialogOpen}>
                      <DialogTrigger asChild>
                        <Button size="icon" variant="outline" data-testid="button-select-rigs">
                          <Plus className="h-4 w-4" />
                        </Button>
                      </DialogTrigger>
                      <DialogContent className="max-w-md">
                        <DialogHeader>
                          <DialogTitle>Select Rigs</DialogTitle>
                          <DialogDescription>
                            Choose one or more rigs to filter your query
                          </DialogDescription>
                        </DialogHeader>
                        <div className="space-y-4">
                          <div className="relative">
                            <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                            <Input
                              placeholder="Search rigs..."
                              value={rigSearchQuery}
                              onChange={(e) => setRigSearchQuery(e.target.value)}
                              className="pl-9"
                              data-testid="input-search-rigs"
                            />
                          </div>
                          <ScrollArea className="h-[300px] border rounded-md">
                            {rigsLoading ? (
                              <div className="p-4 text-center text-muted-foreground">
                                Loading rigs...
                              </div>
                            ) : filteredRigs.length === 0 ? (
                              <div className="p-4 text-center text-muted-foreground">
                                {availableRigs.length === 0 
                                  ? "No rigs found. Upload PDFs to add rigs."
                                  : "No matching rigs found."}
                              </div>
                            ) : (
                              <div className="p-2">
                                {filteredRigs.map((rig, index) => (
                                  <label
                                    key={rig}
                                    className="flex items-center gap-3 p-2 rounded hover-elevate cursor-pointer"
                                    data-testid={`rig-option-${index}`}
                                  >
                                    <Checkbox
                                      checked={selectedRigs.has(rig)}
                                      onCheckedChange={() => toggleRigSelection(rig)}
                                    />
                                    <span className="flex-1">{rig}</span>
                                  </label>
                                ))}
                              </div>
                            )}
                          </ScrollArea>
                          <div className="flex justify-between items-center">
                            <span className="text-sm text-muted-foreground">
                              {selectedRigs.size} selected
                            </span>
                            <Button
                              variant="default"
                              onClick={() => setRigDialogOpen(false)}
                              data-testid="button-close-rig-dialog"
                            >
                              Done
                            </Button>
                          </div>
                        </div>
                      </DialogContent>
                    </Dialog>
                  </div>
                  <p className="text-xs text-muted-foreground">
                    Click + to select from available rigs
                  </p>
                </div>
                <div className="grid grid-cols-2 gap-2">
                  <div className="space-y-2">
                    <Label>Date From</Label>
                    <Input
                      type="date"
                      value={dateFrom}
                      onChange={(e) => setDateFrom(e.target.value)}
                      data-testid="input-date-from"
                    />
                  </div>
                  <div className="space-y-2">
                    <Label>Date To</Label>
                    <Input
                      type="date"
                      value={dateTo}
                      onChange={(e) => setDateTo(e.target.value)}
                      data-testid="input-date-to"
                    />
                  </div>
                </div>
              </div>
            </CardContent>
          </Card>

          <Card className="flex-1 overflow-hidden flex flex-col" data-testid="card-results">
            <CardHeader className="flex-none py-4">
              <div className="flex items-center justify-between gap-2">
                <CardTitle className="text-base">Query Results</CardTitle>
                {queryResult && queryResult.rows.length > 0 && (
                  <Button variant="outline" size="sm" onClick={exportToCsv} data-testid="button-export-csv">
                    <Download className="h-4 w-4 mr-2" />
                    Export CSV
                  </Button>
                )}
              </div>
              {queryResult && (
                <CardDescription>
                  Showing {queryResult.rows.length} of {queryResult.totalRows} records
                </CardDescription>
              )}
            </CardHeader>
            <CardContent className="flex-1 overflow-hidden p-0">
              {executeQueryMutation.isPending ? (
                <div className="flex items-center justify-center h-full">
                  <Loader2 className="h-8 w-8 animate-spin text-primary" />
                </div>
              ) : queryResult && queryResult.rows.length > 0 ? (
                <ScrollArea className="h-full">
                  <div className="p-4">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          {queryResult.columns.map((col) => (
                            <TableHead key={col} className="whitespace-nowrap">
                              {col}
                            </TableHead>
                          ))}
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {queryResult.rows.map((row, i) => (
                          <TableRow key={i} data-testid={`result-row-${i}`}>
                            {queryResult.columns.map((col) => (
                              <TableCell key={col} className="max-w-[200px] truncate">
                                {row[col] ?? "-"}
                              </TableCell>
                            ))}
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </div>
                </ScrollArea>
              ) : (
                <div className="flex flex-col items-center justify-center h-full text-center p-6">
                  <Database className="h-12 w-12 text-muted-foreground/50 mb-4" />
                  <p className="text-muted-foreground">
                    Select tables and columns, then execute your query
                  </p>
                </div>
              )}
            </CardContent>
          </Card>
        </div>

        <Card className="lg:col-span-3 flex flex-col" data-testid="card-actions">
          <CardHeader className="py-4">
            <CardTitle className="text-base">Actions</CardTitle>
          </CardHeader>
          <CardContent className="space-y-3">
            <Button
              className="w-full"
              variant="outline"
              onClick={clearQuery}
              data-testid="button-clear-query"
            >
              <Trash2 className="h-4 w-4 mr-2" />
              Clear Query
            </Button>
            <Button
              className="w-full"
              variant="outline"
              onClick={() => saveQueryMutation.mutate()}
              disabled={selectedColumnsCount === 0 || saveQueryMutation.isPending}
              data-testid="button-save-query"
            >
              {saveQueryMutation.isPending ? (
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
              ) : (
                <Save className="h-4 w-4 mr-2" />
              )}
              Save Query
            </Button>
            <Button
              className="w-full"
              onClick={() => executeQueryMutation.mutate()}
              disabled={selectedColumnsCount === 0 || executeQueryMutation.isPending}
              data-testid="button-execute-query"
            >
              {executeQueryMutation.isPending ? (
                <Loader2 className="h-4 w-4 mr-2 animate-spin" />
              ) : (
                <Play className="h-4 w-4 mr-2" />
              )}
              Execute Query
            </Button>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}
