import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { BarChart3, Download, Filter } from "lucide-react";

interface BitRecord {
  id: number;
  fileId: number;
  bitNumber: string;
  serialNumber: string;
  size: string;
  type: string;
  iadc: string;
  manufacturer: string;
  mdIn: string;
  mdOut: string;
}

export default function RopBitReportPage() {
  const [selectedRig, setSelectedRig] = useState<string>("all");

  const { data: rigs } = useQuery<string[]>({
    queryKey: ["/api/rigs"],
  });

  const { data: bitData, isLoading } = useQuery<{ rows: BitRecord[] }>({
    queryKey: ["/api/query/execute", "bit_data", selectedRig],
    queryFn: async () => {
      const response = await fetch("/api/query/execute", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        credentials: "include",
        body: JSON.stringify({
          tables: {
            bit_data: ["id", "fileId", "bitNumber", "serialNumber", "size", "type", "iadc", "manufacturer", "mdIn", "mdOut"],
          },
          rigName: selectedRig !== "all" ? selectedRig : undefined,
        }),
      });
      return response.json();
    },
  });

  const calculateROP = (mdIn: string, mdOut: string) => {
    const inVal = parseFloat(mdIn) || 0;
    const outVal = parseFloat(mdOut) || 0;
    const footage = outVal - inVal;
    return footage > 0 ? footage.toFixed(1) : "-";
  };

  const handleExportCSV = () => {
    if (!bitData?.rows) return;
    
    const headers = ["Bit #", "Serial #", "Size", "Type", "IADC", "Manufacturer", "MD In", "MD Out", "Footage"];
    const csvData = bitData.rows.map((row) => [
      row.bitNumber,
      row.serialNumber,
      row.size,
      row.type,
      row.iadc,
      row.manufacturer,
      row.mdIn,
      row.mdOut,
      calculateROP(row.mdIn, row.mdOut),
    ]);

    const csv = [headers.join(","), ...csvData.map((row) => row.join(","))].join("\n");
    const blob = new Blob([csv], { type: "text/csv" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = "rop-bit-report.csv";
    a.click();
  };

  return (
    <div className="p-6 h-full overflow-hidden flex flex-col">
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">ROP & Bit Report</h1>
          <p className="text-muted-foreground">Rate of Penetration and Bit Performance Analysis</p>
        </div>
        <div className="flex items-center gap-3">
          <Select value={selectedRig} onValueChange={setSelectedRig}>
            <SelectTrigger className="w-[180px]" data-testid="select-rig">
              <Filter className="h-4 w-4 mr-2" />
              <SelectValue placeholder="Select Rig" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Rigs</SelectItem>
              {rigs?.map((rig) => (
                <SelectItem key={rig} value={rig}>
                  {rig}
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
          <Button variant="outline" onClick={handleExportCSV} data-testid="button-export">
            <Download className="h-4 w-4 mr-2" />
            Export CSV
          </Button>
        </div>
      </div>

      <Card className="flex-1 overflow-hidden flex flex-col">
        <CardHeader className="flex-shrink-0 pb-3 border-b">
          <CardTitle className="text-lg flex items-center gap-2">
            <BarChart3 className="h-5 w-5" />
            Bit Performance Data
            {bitData?.rows && (
              <Badge variant="secondary" className="ml-2">
                {bitData.rows.length} Records
              </Badge>
            )}
          </CardTitle>
        </CardHeader>
        <CardContent className="flex-1 overflow-hidden p-0">
          {isLoading ? (
            <div className="p-4 space-y-3">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-12 w-full" />
              ))}
            </div>
          ) : (
            <ScrollArea className="h-[calc(100vh-300px)]">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Bit #</TableHead>
                    <TableHead>Serial #</TableHead>
                    <TableHead>Size</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>IADC</TableHead>
                    <TableHead>Manufacturer</TableHead>
                    <TableHead className="text-right">MD In (ft)</TableHead>
                    <TableHead className="text-right">MD Out (ft)</TableHead>
                    <TableHead className="text-right">Footage (ft)</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {bitData?.rows && bitData.rows.length > 0 ? (
                    bitData.rows.map((bit, idx) => (
                      <TableRow key={bit.id || idx} data-testid={`row-bit-${idx}`}>
                        <TableCell className="font-mono">{bit.bitNumber || "-"}</TableCell>
                        <TableCell>{bit.serialNumber || "-"}</TableCell>
                        <TableCell>{bit.size || "-"}</TableCell>
                        <TableCell>
                          <Badge variant="outline">{bit.type || "-"}</Badge>
                        </TableCell>
                        <TableCell>{bit.iadc || "-"}</TableCell>
                        <TableCell>{bit.manufacturer || "-"}</TableCell>
                        <TableCell className="text-right font-mono">{bit.mdIn || "-"}</TableCell>
                        <TableCell className="text-right font-mono">{bit.mdOut || "-"}</TableCell>
                        <TableCell className="text-right font-mono font-semibold">
                          {calculateROP(bit.mdIn, bit.mdOut)}
                        </TableCell>
                      </TableRow>
                    ))
                  ) : (
                    <TableRow>
                      <TableCell colSpan={9} className="text-center py-8 text-muted-foreground">
                        <BarChart3 className="h-8 w-8 mx-auto mb-2 opacity-50" />
                        <p>No bit data available</p>
                        <p className="text-sm">Upload and parse PDF files to see bit performance data</p>
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </ScrollArea>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
