import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import { Save, Play, Trash2, Database, AlertCircle } from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Link } from "wouter";

interface SavedQuery {
  id: number;
  userId: string;
  name: string;
  queryConfig: string;
  createdAt: string;
}

export default function SavedReportsPage() {
  const { toast } = useToast();

  const { data: savedQueries, isLoading } = useQuery<SavedQuery[]>({
    queryKey: ["/api/queries/saved"],
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      await apiRequest("DELETE", `/api/queries/saved/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/queries/saved"] });
      toast({
        title: "Query Deleted",
        description: "Saved query has been removed",
      });
    },
    onError: () => {
      toast({
        title: "Delete Failed",
        description: "Failed to delete saved query",
        variant: "destructive",
      });
    },
  });

  const formatDate = (dateStr: string) => {
    return new Date(dateStr).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const parseConfig = (configStr: string) => {
    try {
      const config = JSON.parse(configStr);
      const tables = Object.keys(config.tables || {});
      return {
        tables,
        rigName: config.rigName || "All Rigs",
        dateFrom: config.dateFrom,
        dateTo: config.dateTo,
      };
    } catch {
      return { tables: [], rigName: "Unknown", dateFrom: null, dateTo: null };
    }
  };

  return (
    <div className="p-6 h-full overflow-hidden flex flex-col">
      <div className="flex items-center justify-between mb-6">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-page-title">Saved Reports</h1>
          <p className="text-muted-foreground">View and manage your saved query configurations</p>
        </div>
        <Link href="/query-builder">
          <Button data-testid="button-new-query">
            <Database className="h-4 w-4 mr-2" />
            New Query
          </Button>
        </Link>
      </div>

      <Card className="flex-1 overflow-hidden flex flex-col">
        <CardHeader className="flex-shrink-0 pb-3 border-b">
          <CardTitle className="text-lg flex items-center gap-2">
            <Save className="h-5 w-5" />
            Saved Queries
            {savedQueries && (
              <Badge variant="secondary" className="ml-2">
                {savedQueries.length} Saved
              </Badge>
            )}
          </CardTitle>
        </CardHeader>
        <CardContent className="flex-1 overflow-hidden p-0">
          {isLoading ? (
            <div className="p-4 space-y-3">
              {[1, 2, 3, 4, 5].map((i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : (
            <ScrollArea className="h-[calc(100vh-280px)]">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>Name</TableHead>
                    <TableHead>Tables</TableHead>
                    <TableHead>Rig Filter</TableHead>
                    <TableHead>Date Range</TableHead>
                    <TableHead>Created</TableHead>
                    <TableHead className="text-right">Actions</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {savedQueries && savedQueries.length > 0 ? (
                    savedQueries.map((query) => {
                      const config = parseConfig(query.queryConfig);
                      return (
                        <TableRow key={query.id} data-testid={`row-query-${query.id}`}>
                          <TableCell>
                            <div className="flex items-center gap-2">
                              <Database className="h-4 w-4 text-muted-foreground" />
                              <span className="font-medium">{query.name}</span>
                            </div>
                          </TableCell>
                          <TableCell>
                            <div className="flex flex-wrap gap-1">
                              {config.tables.slice(0, 3).map((table) => (
                                <Badge key={table} variant="outline" className="text-xs">
                                  {table.replace(/_/g, " ")}
                                </Badge>
                              ))}
                              {config.tables.length > 3 && (
                                <Badge variant="secondary" className="text-xs">
                                  +{config.tables.length - 3} more
                                </Badge>
                              )}
                            </div>
                          </TableCell>
                          <TableCell>
                            <Badge variant="secondary">{config.rigName}</Badge>
                          </TableCell>
                          <TableCell className="text-muted-foreground">
                            {config.dateFrom && config.dateTo
                              ? `${config.dateFrom} - ${config.dateTo}`
                              : "All Dates"}
                          </TableCell>
                          <TableCell className="text-sm text-muted-foreground">
                            {formatDate(query.createdAt)}
                          </TableCell>
                          <TableCell className="text-right">
                            <div className="flex items-center justify-end gap-2">
                              <Link href={`/query-builder?load=${query.id}`}>
                                <Button size="sm" variant="outline" data-testid={`button-load-${query.id}`}>
                                  <Play className="h-3 w-3 mr-1" />
                                  Load
                                </Button>
                              </Link>
                              <Button
                                size="sm"
                                variant="ghost"
                                onClick={() => deleteMutation.mutate(query.id)}
                                disabled={deleteMutation.isPending}
                                data-testid={`button-delete-${query.id}`}
                              >
                                <Trash2 className="h-3 w-3 text-destructive" />
                              </Button>
                            </div>
                          </TableCell>
                        </TableRow>
                      );
                    })
                  ) : (
                    <TableRow>
                      <TableCell colSpan={6} className="text-center py-8 text-muted-foreground">
                        <AlertCircle className="h-8 w-8 mx-auto mb-2 opacity-50" />
                        <p>No saved queries yet</p>
                        <p className="text-sm">Create queries in the Query Builder and save them for later</p>
                      </TableCell>
                    </TableRow>
                  )}
                </TableBody>
              </Table>
            </ScrollArea>
          )}
        </CardContent>
      </Card>
    </div>
  );
}
