import { useState, useCallback, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import {
  Upload as UploadIcon,
  FileText,
  CheckCircle,
  XCircle,
  Clock,
  Loader2,
  Trash2,
  Eye,
  RefreshCw,
  Folder,
  File,
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";

interface UploadedFile {
  id: number;
  regName: string;
  secName: string;
  filePath: string;
  uploadDate: string;
  isParsed: boolean;
}

const MONTHS = [
  "January", "February", "March", "April", "May", "June",
  "July", "August", "September", "October", "November", "December"
];

function parseDateFromFilename(filename: string): string | null {
  const match = filename.match(/^[A-Za-z]+(\d{2})(\d{2})(\d{4})\.pdf$/i);
  if (match) {
    const [, month, day, year] = match;
    const monthNum = parseInt(month, 10);
    const dayNum = parseInt(day, 10);
    const yearNum = parseInt(year, 10);
    
    if (monthNum >= 1 && monthNum <= 12 && dayNum >= 1 && dayNum <= 31 && yearNum >= 2000 && yearNum <= 2100) {
      return `${year}-${month}-${day}`;
    }
  }
  return null;
}

export default function UploadPage() {
  const { toast } = useToast();
  const [selectedFile, setSelectedFile] = useState<File | null>(null);
  const [uploadDate, setUploadDate] = useState<string | null>(null);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [isDragging, setIsDragging] = useState(false);
  const [selectedYear, setSelectedYear] = useState<number | null>(null);
  const [selectedMonth, setSelectedMonth] = useState<number | null>(null);
  const [pdfViewerOpen, setPdfViewerOpen] = useState(false);
  const [viewingFile, setViewingFile] = useState<UploadedFile | null>(null);

  const { data: files, isLoading } = useQuery<UploadedFile[]>({
    queryKey: ["/api/files"],
  });

  const filesByYearMonth = useMemo(() => {
    if (!files) return { years: [], monthsByYear: {}, filesByYearMonth: {} };
    
    const years = new Set<number>();
    const monthsByYear: Record<number, Set<number>> = {};
    const grouped: Record<string, UploadedFile[]> = {};
    
    files.forEach(file => {
      const date = new Date(file.uploadDate + "T00:00:00");
      const year = date.getFullYear();
      const month = date.getMonth();
      
      years.add(year);
      if (!monthsByYear[year]) monthsByYear[year] = new Set();
      monthsByYear[year].add(month);
      
      const key = `${year}-${month}`;
      if (!grouped[key]) grouped[key] = [];
      grouped[key].push(file);
    });
    
    return {
      years: Array.from(years).sort((a, b) => b - a),
      monthsByYear: Object.fromEntries(
        Object.entries(monthsByYear).map(([y, m]) => [y, Array.from(m).sort((a, b) => a - b)])
      ),
      filesByYearMonth: grouped
    };
  }, [files]);

  const filteredFiles = useMemo(() => {
    if (selectedYear === null || selectedMonth === null) return [];
    return filesByYearMonth.filesByYearMonth[`${selectedYear}-${selectedMonth}`] || [];
  }, [selectedYear, selectedMonth, filesByYearMonth]);

  const getFilename = (filePath: string) => {
    const match = filePath.match(/([A-Za-z]+\d+\.pdf)$/i);
    if (match) return match[1];
    const lastDash = filePath.lastIndexOf('-');
    return lastDash !== -1 ? filePath.substring(lastDash + 1) : filePath;
  };

  const handleViewPdf = (file: UploadedFile) => {
    setViewingFile(file);
    setPdfViewerOpen(true);
  };

  const uploadMutation = useMutation({
    mutationFn: async (formData: FormData) => {
      const xhr = new XMLHttpRequest();
      
      return new Promise((resolve, reject) => {
        xhr.upload.addEventListener("progress", (e) => {
          if (e.lengthComputable) {
            setUploadProgress(Math.round((e.loaded / e.total) * 100));
          }
        });

        xhr.addEventListener("load", () => {
          if (xhr.status >= 200 && xhr.status < 300) {
            resolve(JSON.parse(xhr.responseText));
          } else {
            reject(new Error(xhr.responseText || "Upload failed"));
          }
        });

        xhr.addEventListener("error", () => reject(new Error("Upload failed")));
        
        xhr.open("POST", "/api/files/upload");
        xhr.send(formData);
      });
    },
    onSuccess: () => {
      toast({
        title: "File uploaded",
        description: "Your PDF is being processed",
      });
      setSelectedFile(null);
      setUploadProgress(0);
      queryClient.invalidateQueries({ queryKey: ["/api/files"] });
      queryClient.invalidateQueries({ queryKey: ["/api/dashboard/stats"] });
    },
    onError: (error) => {
      toast({
        title: "Upload failed",
        description: error.message,
        variant: "destructive",
      });
      setUploadProgress(0);
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (fileId: number) => {
      const response = await fetch(`/api/files/${fileId}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Delete failed");
      return response.json();
    },
    onSuccess: () => {
      toast({
        title: "File deleted",
        description: "The file has been removed",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/files"] });
    },
    onError: () => {
      toast({
        title: "Delete failed",
        description: "Could not delete the file",
        variant: "destructive",
      });
    },
  });

  const handleFileSelect = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file && file.type === "application/pdf") {
      setSelectedFile(file);
      const extractedDate = parseDateFromFilename(file.name);
      if (extractedDate) {
        setUploadDate(extractedDate);
      } else {
        setUploadDate(new Date().toISOString().split("T")[0]);
        toast({
          title: "Date not found in filename",
          description: "Using today's date. Expected format: LMR{MM}{DD}{YYYY}.pdf",
        });
      }
    } else {
      toast({
        title: "Invalid file",
        description: "Please select a PDF file",
        variant: "destructive",
      });
    }
  };

  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(false);
    
    const file = e.dataTransfer.files[0];
    if (file && file.type === "application/pdf") {
      setSelectedFile(file);
      const extractedDate = parseDateFromFilename(file.name);
      if (extractedDate) {
        setUploadDate(extractedDate);
      } else {
        setUploadDate(new Date().toISOString().split("T")[0]);
        toast({
          title: "Date not found in filename",
          description: "Using today's date. Expected format: LMR{MM}{DD}{YYYY}.pdf",
        });
      }
    } else {
      toast({
        title: "Invalid file",
        description: "Please drop a PDF file",
        variant: "destructive",
      });
    }
  }, [toast]);

  const handleUpload = () => {
    if (!selectedFile || !uploadDate) return;
    
    const formData = new FormData();
    formData.append("file", selectedFile);
    formData.append("thedate", uploadDate);
    
    uploadMutation.mutate(formData);
  };

  const handleRemoveFile = () => {
    setSelectedFile(null);
    setUploadDate(null);
  };

  return (
    <div className="flex-1 overflow-auto p-6 space-y-6">
      <div>
        <h1 className="text-2xl font-bold tracking-tight">Upload Files</h1>
        <p className="text-muted-foreground">
          Upload PDF drilling reports to process and extract data
        </p>
      </div>

      <div className="grid lg:grid-cols-2 gap-6">
        <Card data-testid="card-upload">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <UploadIcon className="h-5 w-5" />
              Upload New File
            </CardTitle>
            <CardDescription>
              Drag and drop a PDF file or click to browse
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div
              className={`
                border-2 border-dashed rounded-lg p-8 text-center transition-colors cursor-pointer
                ${isDragging ? "border-primary bg-primary/5" : "border-border hover:border-primary/50"}
                ${selectedFile ? "bg-muted/30" : ""}
              `}
              onDragOver={(e) => { e.preventDefault(); setIsDragging(true); }}
              onDragLeave={() => setIsDragging(false)}
              onDrop={handleDrop}
              onClick={() => document.getElementById("file-input")?.click()}
              data-testid="dropzone"
            >
              <input
                id="file-input"
                type="file"
                accept=".pdf"
                className="hidden"
                onChange={handleFileSelect}
                data-testid="input-file"
              />
              
              {selectedFile ? (
                <div className="flex flex-col items-center gap-2">
                  <FileText className="h-12 w-12 text-primary" />
                  <p className="font-medium">{selectedFile.name}</p>
                  <p className="text-sm text-muted-foreground">
                    {(selectedFile.size / 1024 / 1024).toFixed(2)} MB
                  </p>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={(e) => { e.stopPropagation(); handleRemoveFile(); }}
                    data-testid="button-remove-file"
                  >
                    <XCircle className="h-4 w-4 mr-2" />
                    Remove
                  </Button>
                </div>
              ) : (
                <div className="flex flex-col items-center gap-2">
                  <UploadIcon className="h-12 w-12 text-muted-foreground" />
                  <p className="font-medium">Drop your PDF here</p>
                  <p className="text-sm text-muted-foreground">
                    or click to browse
                  </p>
                </div>
              )}
            </div>

            {uploadDate && (
              <div className="flex items-center gap-2 p-3 bg-muted/50 rounded-lg">
                <Clock className="h-4 w-4 text-muted-foreground" />
                <span className="text-sm text-muted-foreground">Report Date:</span>
                <span className="text-sm font-medium" data-testid="text-upload-date">
                  {new Date(uploadDate + "T00:00:00").toLocaleDateString("en-US", {
                    year: "numeric",
                    month: "long",
                    day: "numeric"
                  })}
                </span>
                <Badge variant="secondary" className="ml-auto">Auto-detected</Badge>
              </div>
            )}

            {uploadProgress > 0 && uploadProgress < 100 && (
              <div className="space-y-2">
                <div className="flex justify-between text-sm">
                  <span>Uploading...</span>
                  <span>{uploadProgress}%</span>
                </div>
                <Progress value={uploadProgress} />
              </div>
            )}

            <Button
              className="w-full"
              onClick={handleUpload}
              disabled={!selectedFile || !uploadDate || uploadMutation.isPending}
              data-testid="button-upload"
            >
              {uploadMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Processing...
                </>
              ) : (
                <>
                  <UploadIcon className="h-4 w-4 mr-2" />
                  Upload & Process
                </>
              )}
            </Button>
          </CardContent>
        </Card>

        <Card data-testid="card-info">
          <CardHeader>
            <CardTitle>Supported Format</CardTitle>
            <CardDescription>
              Information about file requirements
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-3">
              <div className="flex items-start gap-3">
                <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
                <div>
                  <p className="font-medium">PDF Drilling Reports</p>
                  <p className="text-sm text-muted-foreground">
                    Standard format drilling daily reports in PDF
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
                <div>
                  <p className="font-medium">Multi-page Support</p>
                  <p className="text-sm text-muted-foreground">
                    Process multiple reports in a single PDF file
                  </p>
                </div>
              </div>
              <div className="flex items-start gap-3">
                <CheckCircle className="h-5 w-5 text-green-500 mt-0.5" />
                <div>
                  <p className="font-medium">Automatic Parsing</p>
                  <p className="text-sm text-muted-foreground">
                    Data is extracted and stored in the database automatically
                  </p>
                </div>
              </div>
            </div>

            <div className="border-t border-border pt-4">
              <h4 className="font-medium mb-2">Extracted Data Includes:</h4>
              <div className="flex flex-wrap gap-2">
                <Badge variant="secondary">Rig Details</Badge>
                <Badge variant="secondary">Bit Data</Badge>
                <Badge variant="secondary">Drill String</Badge>
                <Badge variant="secondary">Mud Data</Badge>
                <Badge variant="secondary">Companies</Badge>
                <Badge variant="secondary">Lost Time</Badge>
                <Badge variant="secondary">Survey Data</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Card data-testid="card-files-list">
        <CardHeader>
          <div className="flex items-center justify-between">
            <div>
              <CardTitle>Files Table</CardTitle>
              <CardDescription>
                {files?.length || 0} files in the system
              </CardDescription>
            </div>
            <Button
              variant="outline"
              size="sm"
              onClick={() => queryClient.invalidateQueries({ queryKey: ["/api/files"] })}
              data-testid="button-refresh-files"
            >
              <RefreshCw className="h-4 w-4 mr-2" />
              Refresh
            </Button>
          </div>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="h-8 w-8 animate-spin text-primary" />
            </div>
          ) : files && files.length > 0 ? (
            <div className="grid lg:grid-cols-12 gap-6">
              <div className="lg:col-span-2">
                <Card data-testid="card-years">
                  <CardHeader className="py-3 px-4">
                    <CardTitle className="text-sm">Years</CardTitle>
                  </CardHeader>
                  <CardContent className="p-2 overflow-hidden">
                    <ScrollArea className="h-[350px]">
                      <div className="space-y-1">
                        {filesByYearMonth.years.map(year => (
                          <button
                            key={year}
                            onClick={() => {
                              setSelectedYear(year);
                              setSelectedMonth(null);
                            }}
                            className={`w-full flex items-center gap-2 px-3 py-2 rounded-md text-sm transition-colors ${
                              selectedYear === year 
                                ? "bg-primary text-primary-foreground" 
                                : "hover-elevate"
                            }`}
                            data-testid={`button-year-${year}`}
                          >
                            <Folder className="h-4 w-4" />
                            {year}
                          </button>
                        ))}
                      </div>
                    </ScrollArea>
                  </CardContent>
                </Card>
              </div>

              <div className="lg:col-span-3">
                {selectedYear && (
                  <>
                    <h3 className="text-lg font-semibold mb-3" data-testid="text-selected-year">{selectedYear}</h3>
                    <ScrollArea className="h-[350px]">
                      <div className="grid grid-cols-2 gap-2 pr-3">
                        {(filesByYearMonth.monthsByYear[selectedYear] || []).map(monthIdx => (
                          <button
                            key={monthIdx}
                            onClick={() => setSelectedMonth(monthIdx)}
                            className={`flex flex-col items-center gap-1 p-4 rounded-lg border transition-colors ${
                              selectedMonth === monthIdx 
                                ? "border-primary bg-primary/10" 
                                : "border-border hover-elevate"
                            }`}
                            data-testid={`button-month-${monthIdx}`}
                          >
                            <File className="h-8 w-8 text-primary" />
                            <span className="text-sm font-medium">{MONTHS[monthIdx]}</span>
                          </button>
                        ))}
                      </div>
                    </ScrollArea>
                  </>
                )}
                {!selectedYear && (
                  <div className="flex flex-col items-center justify-center py-8 text-center text-muted-foreground">
                    <Folder className="h-10 w-10 mb-2 opacity-50" />
                    <p className="text-sm">Select a year to view months</p>
                  </div>
                )}
              </div>

              <div className="lg:col-span-7">
                {selectedYear && selectedMonth !== null && filteredFiles.length > 0 ? (
                  <ScrollArea className="h-[400px]">
                    <Table>
                      <TableHeader>
                        <TableRow>
                          <TableHead>File Name</TableHead>
                          <TableHead>Date</TableHead>
                          <TableHead>Status</TableHead>
                          <TableHead className="text-right">Actions</TableHead>
                        </TableRow>
                      </TableHeader>
                      <TableBody>
                        {filteredFiles.map((file) => (
                          <TableRow key={file.id} data-testid={`file-row-${file.id}`}>
                            <TableCell className="font-medium">
                              {getFilename(file.filePath)}
                            </TableCell>
                            <TableCell>{file.uploadDate}</TableCell>
                            <TableCell>
                              {file.isParsed ? (
                                <Badge className="bg-green-500/10 text-green-600 hover:bg-green-500/20">
                                  <CheckCircle className="h-3 w-3 mr-1" />
                                  Parsed
                                </Badge>
                              ) : (
                                <Badge variant="secondary">
                                  <Clock className="h-3 w-3 mr-1" />
                                  Pending
                                </Badge>
                              )}
                            </TableCell>
                            <TableCell className="text-right">
                              <div className="flex items-center justify-end gap-1">
                                <Button 
                                  variant="ghost" 
                                  size="sm"
                                  onClick={() => handleViewPdf(file)}
                                  data-testid={`button-view-${file.id}`}
                                >
                                  <Eye className="h-4 w-4 mr-1" />
                                  Show
                                </Button>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => deleteMutation.mutate(file.id)}
                                  disabled={deleteMutation.isPending}
                                  data-testid={`button-delete-${file.id}`}
                                >
                                  <Trash2 className="h-4 w-4" />
                                </Button>
                              </div>
                            </TableCell>
                          </TableRow>
                        ))}
                      </TableBody>
                    </Table>
                  </ScrollArea>
                ) : selectedYear && selectedMonth !== null ? (
                  <div className="flex flex-col items-center justify-center py-12 text-center">
                    <FileText className="h-12 w-12 text-muted-foreground/50 mb-4" />
                    <p className="text-muted-foreground">No files for this month</p>
                  </div>
                ) : (
                  <div className="flex flex-col items-center justify-center py-12 text-center">
                    <FileText className="h-12 w-12 text-muted-foreground/50 mb-4" />
                    <p className="text-muted-foreground">Select a year and month to view files</p>
                  </div>
                )}
              </div>
            </div>
          ) : (
            <div className="flex flex-col items-center justify-center py-12 text-center">
              <FileText className="h-12 w-12 text-muted-foreground/50 mb-4" />
              <p className="text-muted-foreground">No files uploaded yet</p>
              <p className="text-sm text-muted-foreground">
                Upload a PDF to get started
              </p>
            </div>
          )}
        </CardContent>
      </Card>

      <Dialog open={pdfViewerOpen} onOpenChange={setPdfViewerOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh]">
          <DialogHeader>
            <DialogTitle>
              {viewingFile ? getFilename(viewingFile.filePath) : "PDF Viewer"}
            </DialogTitle>
          </DialogHeader>
          {viewingFile && (
            <iframe
              src={`/uploads/${viewingFile.filePath}`}
              className="w-full h-[70vh] border rounded-md"
              title="PDF Viewer"
            />
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
}
