import { db } from "./db";
import {
  multipleFiles,
  firstTable,
  secondTable,
  thirdTable,
  companies,
  bitData,
  drillString,
  drillStringFloats,
  mudData,
  mudTreatment,
  lostTimeDetails,
  miscellaneous,
  trucks,
  bulks,
  repairTable,
  outOfScope,
  otherDataTable,
} from "@shared/schema";

function extractBetween(text: string, start: string, end: string): string {
  const startEsc = start.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
  const endEsc = end.replace(/[.*+?^${}()|[\]\\]/g, "\\$&");
  const pattern = new RegExp(startEsc + "([\\s\\S]*?)" + endEsc);
  const match = text.match(pattern);
  return match ? match[1].replace(/\n/g, " ").replace(/"""/g, "").trim() : "";
}

function cleanText(text: string): string {
  return text.replace(/\n/g, " ").replace(/"""/g, "").trim();
}

export async function parsePdfContent(
  pdfText: string,
  fileName: string,
  uploadDate: string
): Promise<{ fileCount: number; rigNames: string[] }> {
  const year = new Date(uploadDate).getFullYear().toString();
  const copyrightPattern = new RegExp(
    `Copyright ${year} Saudi Aramco. All Rights Reserved`,
    "g"
  );
  let cleanedText = pdfText.replace(copyrightPattern, "");

  const pagePattern = /Page\s+\d+\s+of\s+\d+[\s\S]*?\d{2}\/\d{2}\/\d{4}\s+-\s+\d{2}\/\d{2}\/\d{4}/g;
  cleanedText = cleanedText.replace(pagePattern, "");

  const reports = cleanedText.split("Foreman(s)");
  const rigNames: string[] = [];
  let fileCount = 0;

  for (let i = 1; i < reports.length; i++) {
    let reportText = reports[i];

    try {
      const date = extractBetween(reportText, "Date: ", "Rig");
      if (!date) continue;

      const rigSection = extractBetween(reportText, date, "Well");
      const rigMatch = rigSection.match(/Rig\s*(.+)/i);
      const regName = rigMatch ? rigMatch[1].trim() : "Unknown Rig";

      const secName = extractBetween(reportText, "Well", "Charge");
      const firstBlock = extractBetween(reportText, "Well", "Prev. Depth (ft)");

      const [file] = await db
        .insert(multipleFiles)
        .values({
          regName: regName,
          secName: secName,
          filePath: fileName,
          uploadDate: uploadDate,
          isParsed: true,
        })
        .returning();

      const fileId = file.id;
      fileCount++;

      if (!rigNames.includes(regName)) {
        rigNames.push(regName);
      }

      const companiesText = extractBetween(
        reportText,
        "MISCELLANEOUS\n",
        "TRUCKS/BOATS"
      );
      if (companiesText) {
        await parseCompaniesTable(companiesText, fileId);
      }

      const wellNoAndCharge = extractBetween(reportText, "Charge #:", "Wellbores");
      const wellbores = extractBetween(firstBlock, "Wellbores:", "Objective :");
      const objective = extractBetween(firstBlock, "Objective :", "THURAYA");

      let foremanEngineer = "";
      const engineerPos = reportText.indexOf("Engineer");
      if (engineerPos !== -1) {
        foremanEngineer = cleanText(reportText.substring(0, engineerPos));
      }

      const engineer = extractBetween(reportText, "Engineer", "Manager");
      const manager = extractBetween(
        reportText,
        "Manager ",
        " Morning Report (Service Providers View)"
      );
      const thuraya = extractBetween(reportText, "THURAYA", "RIG FORMAN VSAT");
      const rigFormanVsat = extractBetween(
        reportText,
        "RIG FORMAN VSAT",
        "Last 24 hr operations"
      );

      let rigFormanVsatClean = "";
      let contractorClerk = "";
      if (rigFormanVsat) {
        const parts = rigFormanVsat.split("CONTRACTOR /CLERK VSA");
        rigFormanVsatClean = parts[0] || "";
        contractorClerk = parts[1] || "";
      }

      await db.insert(firstTable).values({
        fileId,
        date,
        wellNoAndCharge,
        wellbores,
        objective,
        rig: regName,
        foremanEngineer,
        manager,
        engineer,
        thurayaRigForman: thuraya,
        rigFormanVsat: rigFormanVsatClean,
        contractorClerk,
      });

      const last24HrOps = extractBetween(
        reportText,
        "Last 24 hr operations",
        "Next 24 hr plan"
      );
      const next24HrPlan = extractBetween(
        reportText,
        "Next 24 hr plan",
        "Location"
      );
      const location = extractBetween(reportText, "Location", "Next Location");
      const nextLocation = extractBetween(
        reportText,
        "Next Location",
        "Current Depth (ft)"
      );
      const depth = extractBetween(
        reportText,
        "Current Depth (ft)",
        "Last Csg Size"
      );
      const linerSizeSection = extractBetween(
        reportText,
        "Size",
        "Prev. Depth (ft)"
      );
      const linerParts = linerSizeSection.split("Liner Size");
      const lastCsgSize = linerParts[0] || "";
      const linerSize = linerParts[1] || "";

      const prevDepth = extractBetween(
        reportText,
        "Prev. Depth (ft)",
        "Days Since Spud/Comm"
      );
      const daysSinceSpud = extractBetween(reportText, "(Date)", "Circ %");
      const circ = extractBetween(reportText, "Circ %", "Footage (ft)");
      const footage = extractBetween(
        reportText,
        "Footage (ft)",
        "Distance from DHA M/Gate (KM)"
      );
      const distanceFromDha = extractBetween(
        reportText,
        "Distance from DHA M/Gate (KM)",
        "Formation Tops"
      );
      const formationTops = extractBetween(
        reportText,
        "Formation Tops",
        "Depth Comments"
      );
      const depthComments = extractBetween(
        reportText,
        "Depth Comments",
        "Hole depth Event depth"
      );

      await db.insert(secondTable).values({
        fileId,
        depth,
        last24HrOperations: last24HrOps,
        next24HrPlan,
        location,
        nextLocation,
        daysSinceSpudAndDate: daysSinceSpud,
        formationTops,
        depthComments,
        linerSizeTolMdTvd: linerSize,
        lastCsgSizeMdTvd: lastCsgSize,
        footage,
        distanceFromDha,
        circ,
      });

      const hasLostTime = reportText.includes("Lost Time Details");
      const hasOutOfScope = reportText.includes("Out Of Scope Details");

      let thirdTableText = "";
      if (hasLostTime) {
        thirdTableText = extractBetween(
          reportText,
          "Summary of Operations",
          "Lost Time Details"
        );
      } else if (hasOutOfScope) {
        thirdTableText = extractBetween(
          reportText,
          "Summary of Operations",
          "Out Of Scope Details"
        );
      } else {
        thirdTableText = extractBetween(
          reportText,
          "Summary of Operations",
          "Foreman Remarks"
        );
      }

      if (thirdTableText) {
        await parseThirdTable(thirdTableText, fileId);
      }

      if (hasLostTime) {
        const lostTimeText = extractBetween(
          reportText,
          "Lost Time Details",
          hasOutOfScope ? "Out Of Scope Details" : "Foreman Remarks"
        );
        if (lostTimeText) {
          await parseLostTimeTable(lostTimeText, fileId);
        }
      }

      if (hasOutOfScope) {
        const outOfScopeText = extractBetween(
          reportText,
          "Out Of Scope Details",
          "Foreman Remarks"
        );
        if (outOfScopeText) {
          await parseOutOfScopeTable(outOfScopeText, fileId);
        }
      }

      const foremanRemarks = extractBetween(
        reportText,
        "Foreman Remarks",
        "MUD DATA"
      );
      if (foremanRemarks) {
        await db.insert(otherDataTable).values({
          fileId,
          comment: foremanRemarks,
        });
      }

      const mudDataText = extractBetween(reportText, "MUD DATA", "MUD TREATMENT");
      if (mudDataText) {
        await parseMudData(mudDataText, fileId);
      }

      const mudTreatmentText = extractBetween(
        reportText,
        "MUD TREATMENT",
        "BIT DATA"
      );
      if (mudTreatmentText) {
        await parseMudTreatment(mudTreatmentText, fileId);
      }

      const bitDataText = extractBetween(reportText, "BIT DATA", "DRILL STRING");
      if (bitDataText) {
        await parseBitData(bitDataText, fileId);
      }

      const drillStringText = extractBetween(
        reportText,
        "DRILL STRING",
        "DRILL STRING FLOATS"
      );
      if (drillStringText) {
        await parseDrillString(drillStringText, fileId);
      }

      const drillStringFloatsText = extractBetween(
        reportText,
        "DRILL STRING FLOATS",
        "MISCELLANEOUS"
      );
      if (drillStringFloatsText) {
        await parseDrillStringFloats(drillStringFloatsText, fileId);
      }

      const miscText = extractBetween(
        reportText,
        "MISCELLANEOUS",
        "TRUCKS/BOATS"
      );
      if (miscText) {
        await parseMiscellaneous(miscText, fileId);
      }

      const trucksText = extractBetween(
        reportText,
        "TRUCKS/BOATS",
        "BULKS USED"
      );
      if (trucksText) {
        await parseTrucks(trucksText, fileId);
      }

      const bulksText = extractBetween(reportText, "BULKS USED", "REPAIR TABLE");
      if (bulksText) {
        await parseBulks(bulksText, fileId);
      }

      const repairText = extractBetween(reportText, "REPAIR TABLE", "END");
      if (repairText) {
        await parseRepairTable(repairText, fileId);
      }
    } catch (error) {
      console.error(`Error parsing report ${i}:`, error);
    }
  }

  return { fileCount, rigNames };
}

async function parseCompaniesTable(text: string, fileId: number) {
  const cleanedText = text
    .replace(/Weight[\s\S]*?Operating Hours\./g, "")
    .replace(/PERSONNEL[\s\S]*?INFORMATION/g, "")
    .replace(/Company/g, "")
    .replace(/Category/g, "")
    .replace(/Personnel on Loc\. Hrs\./g, "")
    .replace(/Operating Hours\./g, "")
    .replace(/\n/g, " ")
    .trim();

  const parts = cleanedText.split("Total");
  if (parts.length < 2) return;

  const dataText = parts[0].trim();
  const totalText = parts[1].trim();
  const words = dataText.split(/\s+/).filter((w) => w);
  const totalWords = totalText.split(/\s+/).filter((w) => w);
  const total = totalWords[0] || "";

  for (let j = 0; j < words.length; j += 5) {
    if (j + 4 < words.length) {
      try {
        await db.insert(companies).values({
          fileId,
          company: words[j] || "",
          category: words[j + 1] || "",
          noOfPersons: words[j + 2] || "",
          onLocationHrs: words[j + 3] || "",
          operatingHrs: words[j + 4] || "",
          total,
        });
      } catch (e) {
        console.error("Error inserting company:", e);
      }
    }
  }
}

async function parseThirdTable(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  let currentData: any = null;
  let summaryOfOps = "";

  for (const row of rows) {
    const trimmedRow = row.trim();
    if (!trimmedRow) continue;

    const fromToMatch = trimmedRow.match(/^(\d{4}-\d{4})\s*([\d.]+)/);
    if (fromToMatch) {
      if (currentData) {
        currentData.summaryOfOperations = summaryOfOps.trim();
        try {
          await db.insert(thirdTable).values(currentData);
        } catch (e) {
          console.error("Error inserting third table:", e);
        }
      }

      currentData = {
        fileId,
        fromTo: fromToMatch[1],
        hrs: fromToMatch[2],
        lateral: "",
        holeSize: "",
        cat: "",
        majorOp: "",
        action: "",
        object: "",
        respCo: "",
        holeDepth: "",
        eventDepth: "",
        summaryOfOperations: "",
        ltType: "",
        ltId: "",
      };
      summaryOfOps = "";

      const remaining = trimmedRow.substring(fromToMatch[0].length).trim();
      const parts = remaining.split(/\s+/);
      if (parts.length >= 8) {
        currentData.lateral = parts[0] || "";
        currentData.holeSize = parts[1] || "";
        currentData.cat = parts[2] || "";
        currentData.majorOp = parts[3] || "";
        currentData.action = parts[4] || "";
        currentData.object = parts[5] || "";
        currentData.respCo = parts[6] || "";

        const depthParts = parts.slice(7);
        const numbers = depthParts.filter((p) => /^\d+$/.test(p));
        if (numbers.length >= 2) {
          currentData.holeDepth = `${numbers[0]}-${numbers[1]}`;
          if (numbers.length >= 4) {
            currentData.eventDepth = `${numbers[2]}-${numbers[3]}`;
          }
        }
      }
    } else if (currentData) {
      summaryOfOps += " " + trimmedRow;
    }
  }

  if (currentData) {
    currentData.summaryOfOperations = summaryOfOps.trim();
    try {
      await db.insert(thirdTable).values(currentData);
    } catch (e) {
      console.error("Error inserting final third table row:", e);
    }
  }
}

async function parseLostTimeTable(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 4) {
      try {
        await db.insert(lostTimeDetails).values({
          fileId,
          ltType: parts[0] || "",
          hrs: parts[1] || "",
          ltCode: parts[2] || "",
          ltDesc: parts.slice(3).join(" ") || "",
        });
      } catch (e) {
        console.error("Error inserting lost time:", e);
      }
    }
  }
}

async function parseOutOfScopeTable(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 2) {
      try {
        await db.insert(outOfScope).values({
          fileId,
          oosType: parts[0] || "",
          hrs: parts[1] || "",
          oosSummary: parts.slice(2).join(" ") || "",
        });
      } catch (e) {
        console.error("Error inserting out of scope:", e);
      }
    }
  }
}

async function parseMudData(text: string, fileId: number) {
  const cleanedText = text.replace(/\n/g, " ").trim();
  const parts = cleanedText.split(/\s+/);

  if (parts.length >= 5) {
    try {
      await db.insert(mudData).values({
        fileId,
        weightPcf: parts[0] || "",
        funnelSec: parts[1] || "",
        pv: parts[2] || "",
        yp: parts[3] || "",
        filtrateWlApi: parts[4] || "",
      });
    } catch (e) {
      console.error("Error inserting mud data:", e);
    }
  }
}

async function parseMudTreatment(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 3) {
      try {
        await db.insert(mudTreatment).values({
          fileId,
          material: parts[0] || "",
          quantity: parts[1] || "",
          unit: parts[2] || "",
        });
      } catch (e) {
        console.error("Error inserting mud treatment:", e);
      }
    }
  }
}

async function parseBitData(text: string, fileId: number) {
  const cleanedText = text.replace(/\n/g, " ").trim();
  const parts = cleanedText.split(/\s+/);

  if (parts.length >= 5) {
    try {
      await db.insert(bitData).values({
        fileId,
        bitNumber: parts[0] || "",
        serialNumber: parts[1] || "",
        size: parts[2] || "",
        type: parts[3] || "",
        iadc: parts[4] || "",
        manufacturer: parts[5] || "",
        mdIn: parts[6] || "",
        mdOut: parts[7] || "",
      });
    } catch (e) {
      console.error("Error inserting bit data:", e);
    }
  }
}

async function parseDrillString(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 5) {
      try {
        await db.insert(drillString).values({
          fileId,
          component: parts[0] || "",
          odSize: parts[1] || "",
          idSize: parts[2] || "",
          length: parts[3] || "",
          provider: parts[4] || "",
        });
      } catch (e) {
        console.error("Error inserting drill string:", e);
      }
    }
  }
}

async function parseDrillStringFloats(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 3) {
      try {
        await db.insert(drillStringFloats).values({
          fileId,
          float: parts[0] || "",
          bhaHrs: parts[1] || "",
          stringWt: parts[2] || "",
        });
      } catch (e) {
        console.error("Error inserting drill string floats:", e);
      }
    }
  }
}

async function parseMiscellaneous(text: string, fileId: number) {
  const cleanedText = text.replace(/\n/g, " ").trim();
  const parts = cleanedText.split(/\s+/);

  try {
    await db.insert(miscellaneous).values({
      fileId,
      bopTest: parts[0] || "",
      bopDrills: parts[1] || "",
      wind: parts[2] || "",
      sea: parts[3] || "",
      weather: parts[4] || "",
    });
  } catch (e) {
    console.error("Error inserting miscellaneous:", e);
  }
}

async function parseTrucks(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    if (row.trim()) {
      try {
        await db.insert(trucks).values({
          fileId,
          truck: row.trim(),
        });
      } catch (e) {
        console.error("Error inserting truck:", e);
      }
    }
  }
}

async function parseBulks(text: string, fileId: number) {
  const rows = text.split("\n").filter((r) => r.trim());
  for (const row of rows) {
    const parts = row.trim().split(/\s+/);
    if (parts.length >= 2) {
      try {
        await db.insert(bulks).values({
          fileId,
          bulk: parts[0] || "",
          value: parts[1] || "",
        });
      } catch (e) {
        console.error("Error inserting bulk:", e);
      }
    }
  }
}

async function parseRepairTable(text: string, fileId: number) {
  const cleanedText = text.replace(/\n/g, " ").trim();
  const parts = cleanedText.split(/\s+/);

  if (parts.length >= 2) {
    try {
      await db.insert(repairTable).values({
        fileId,
        instrumentation: parts[0] || "",
        other: parts[1] || "",
        reporter: parts[2] || "",
        computer: parts[3] || "",
      });
    } catch (e) {
      console.error("Error inserting repair table:", e);
    }
  }
}
