import type { Express } from "express";
import express from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { setupAuth, registerAuthRoutes, isAuthenticated } from "./replit_integrations/auth";
import multer from "multer";
import path from "path";
import fs from "fs";
import OpenAI from "openai";
import { PDFParse } from "pdf-parse";
import { parsePdfContent } from "./pdfParser";

const upload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => {
      const uploadDir = path.join(process.cwd(), "uploads");
      if (!fs.existsSync(uploadDir)) {
        fs.mkdirSync(uploadDir, { recursive: true });
      }
      cb(null, uploadDir);
    },
    filename: (req, file, cb) => {
      const uniqueSuffix = Date.now() + "-" + Math.round(Math.random() * 1e9);
      cb(null, uniqueSuffix + "-" + file.originalname);
    },
  }),
  fileFilter: (req, file, cb) => {
    if (file.mimetype === "application/pdf") {
      cb(null, true);
    } else {
      cb(new Error("Only PDF files are allowed"));
    }
  },
});

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  await setupAuth(app);
  registerAuthRoutes(app);

  const uploadsDir = path.join(process.cwd(), "uploads");
  if (!fs.existsSync(uploadsDir)) {
    fs.mkdirSync(uploadsDir, { recursive: true });
  }
  app.use("/uploads", isAuthenticated, express.static(uploadsDir));

  app.get("/api/dashboard/stats", isAuthenticated, async (req, res) => {
    try {
      const stats = await storage.getDashboardStats();
      res.json(stats);
    } catch (error) {
      console.error("Error fetching dashboard stats:", error);
      res.status(500).json({ message: "Failed to fetch stats" });
    }
  });

  app.get("/api/dashboard/tables", isAuthenticated, async (req, res) => {
    try {
      const stats = await storage.getTableStats();
      res.json(stats);
    } catch (error) {
      console.error("Error fetching table stats:", error);
      res.status(500).json({ message: "Failed to fetch table stats" });
    }
  });

  app.get("/api/dashboard/chart", isAuthenticated, async (req, res) => {
    res.json([
      { date: "Jan", uploads: 12, records: 340 },
      { date: "Feb", uploads: 19, records: 520 },
      { date: "Mar", uploads: 15, records: 410 },
      { date: "Apr", uploads: 25, records: 680 },
      { date: "May", uploads: 22, records: 590 },
      { date: "Jun", uploads: 30, records: 820 },
    ]);
  });

  app.get("/api/files", isAuthenticated, async (req, res) => {
    try {
      const files = await storage.getFiles();
      res.json(files);
    } catch (error) {
      console.error("Error fetching files:", error);
      res.status(500).json({ message: "Failed to fetch files" });
    }
  });

  app.get("/api/files/recent", isAuthenticated, async (req, res) => {
    try {
      const files = await storage.getRecentFiles(5);
      res.json(files);
    } catch (error) {
      console.error("Error fetching recent files:", error);
      res.status(500).json({ message: "Failed to fetch files" });
    }
  });

  app.post("/api/files/upload", isAuthenticated, upload.single("file"), async (req: any, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ message: "No file uploaded" });
      }

      const uploadDate = req.body.thedate || new Date().toISOString().split("T")[0];
      
      const file = await storage.createFile({
        regName: "Multiple Rigs",
        secName: "Daily Report",
        filePath: req.file.filename,
        uploadDate,
        isParsed: false,
      });

      try {
        const pdfBuffer = fs.readFileSync(req.file.path);
        const parser = new PDFParse({ data: pdfBuffer });
        const pdfData = await parser.getText();
        const pdfText = pdfData.text;
        
        const result = await parsePdfContent(
          pdfText,
          req.file.filename,
          uploadDate
        );
        
        await storage.updateFileParsed(file.id, true);
        
        res.json({ 
          ...file, 
          isParsed: true,
          message: `Parsed ${result.fileCount} reports from PDF`,
          reportCount: result.fileCount,
          rigNames: result.rigNames,
        });
      } catch (parseError) {
        console.error("Error parsing PDF:", parseError);
        res.json({ ...file, parseError: "PDF parsing failed, can retry manually" });
      }
    } catch (error) {
      console.error("Error uploading file:", error);
      res.status(500).json({ message: "Failed to upload file" });
    }
  });

  app.delete("/api/files/:id", isAuthenticated, async (req, res) => {
    try {
      const id = parseInt(req.params.id as string);
      await storage.deleteFile(id);
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting file:", error);
      res.status(500).json({ message: "Failed to delete file" });
    }
  });

  app.post("/api/files/:id/parse", isAuthenticated, async (req, res) => {
    try {
      const id = parseInt(req.params.id as string);
      const file = await storage.getFileById(id);
      
      if (!file) {
        return res.status(404).json({ message: "File not found" });
      }

      const pdfPath = path.join(process.cwd(), "uploads", file.filePath);
      
      if (!fs.existsSync(pdfPath)) {
        return res.status(404).json({ message: "PDF file not found on disk" });
      }

      const pdfBuffer = fs.readFileSync(pdfPath);
      const parser = new PDFParse({ data: pdfBuffer });
      const pdfData = await parser.getText();
      const pdfText = pdfData.text;

      const result = await parsePdfContent(
        pdfText,
        file.filePath,
        file.uploadDate || new Date().toISOString().split("T")[0]
      );

      await storage.updateFileParsed(id, true);

      res.json({
        success: true,
        message: `Parsed ${result.fileCount} reports from PDF`,
        reportCount: result.fileCount,
        rigNames: result.rigNames,
      });
    } catch (error) {
      console.error("Error parsing file:", error);
      res.status(500).json({ message: "Failed to parse file" });
    }
  });

  app.get("/api/rigs", isAuthenticated, async (req, res) => {
    try {
      const rigNames = await storage.getUniqueRigNames();
      res.json(rigNames);
    } catch (error) {
      console.error("Error fetching rig names:", error);
      res.status(500).json({ message: "Failed to fetch rig names" });
    }
  });

  app.post("/api/query/execute", isAuthenticated, async (req, res) => {
    try {
      const { tables, rigName, dateFrom, dateTo } = req.body;
      
      if (!tables || Object.keys(tables).length === 0) {
        return res.status(400).json({ message: "No tables selected" });
      }

      const result = await storage.executeQuery({ tables, rigName, dateFrom, dateTo });
      res.json(result);
    } catch (error) {
      console.error("Error executing query:", error);
      res.status(500).json({ message: "Failed to execute query" });
    }
  });

  app.post("/api/query/save", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const { name, config } = req.body;
      const query = await storage.createSavedQuery({
        userId,
        name,
        queryConfig: JSON.stringify(config),
      });
      res.json(query);
    } catch (error) {
      console.error("Error saving query:", error);
      res.status(500).json({ message: "Failed to save query" });
    }
  });

  app.get("/api/queries/saved", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const queries = await storage.getSavedQueries(userId);
      res.json(queries);
    } catch (error) {
      console.error("Error fetching saved queries:", error);
      res.status(500).json({ message: "Failed to fetch saved queries" });
    }
  });

  app.delete("/api/queries/saved/:id", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const id = parseInt(req.params.id);
      await storage.deleteSavedQuery(id, userId);
      res.json({ success: true });
    } catch (error) {
      console.error("Error deleting saved query:", error);
      res.status(500).json({ message: "Failed to delete saved query" });
    }
  });

  app.post("/api/query/export", isAuthenticated, async (req, res) => {
    try {
      const { tables, rigName, dateFrom, dateTo } = req.body;
      
      if (!tables || Object.keys(tables).length === 0) {
        return res.status(400).json({ message: "No tables selected" });
      }

      const result = await storage.executeQuery({ tables, rigName, dateFrom, dateTo });
      
      if (result.rows.length === 0) {
        return res.status(400).json({ message: "No data to export" });
      }

      const escapeCsvValue = (value: any): string => {
        if (value === null || value === undefined) return "";
        const str = String(value);
        if (str.includes(",") || str.includes('"') || str.includes("\n")) {
          return `"${str.replace(/"/g, '""')}"`;
        }
        return str;
      };

      const csvRows: string[] = [];
      csvRows.push(result.columns.map(escapeCsvValue).join(","));
      
      for (const row of result.rows) {
        const values = result.columns.map((col) => escapeCsvValue(row[col]));
        csvRows.push(values.join(","));
      }

      const csv = csvRows.join("\n");
      
      res.setHeader("Content-Type", "text/csv");
      res.setHeader("Content-Disposition", `attachment; filename="arms-export-${Date.now()}.csv"`);
      res.send(csv);
    } catch (error) {
      console.error("Error exporting query:", error);
      res.status(500).json({ message: "Failed to export data" });
    }
  });

  app.get("/api/chat/messages", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const messages = await storage.getChatMessages(userId);
      res.json(messages);
    } catch (error) {
      console.error("Error fetching messages:", error);
      res.status(500).json({ message: "Failed to fetch messages" });
    }
  });

  app.post("/api/chat/send", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      const { content } = req.body;
      if (!content) {
        return res.status(400).json({ message: "Message content required" });
      }

      await storage.createChatMessage({
        userId,
        role: "user",
        content,
      });

      let assistantResponse = "";
      
      try {
        const openai = new OpenAI({
          apiKey: process.env.AI_INTEGRATIONS_OPENAI_API_KEY,
          baseURL: process.env.AI_INTEGRATIONS_OPENAI_BASE_URL,
        });
        
        const stats = await storage.getDashboardStats();
        const tableStats = await storage.getTableStats();
        
        const systemPrompt = `You are ARMS AI Assistant, helping users analyze drilling report data. 
You have access to the following data summary:
- Total files: ${stats.totalFiles}
- Total records: ${stats.totalRecords}
- Recent uploads (7 days): ${stats.recentUploads}
- Table distribution: ${tableStats.map((t) => `${t.name}: ${t.count} records`).join(", ")}

The system contains drilling data including:
- Rig Details (dates, well numbers, objectives, personnel)
- Bit Data (bit numbers, manufacturers, drilling metrics)
- Drill String (components, sizes, weights)
- Mud Data (weights, viscosity, chemistry)
- Companies (service providers, personnel counts)
- Lost Time (causes, durations, impacts)
- Survey Data (coordinates, angles, depths)

Help users understand their drilling data by answering questions, suggesting analyses, and providing insights.
Keep responses concise and focused on the data available in the system.`;

        const completion = await openai.chat.completions.create({
          model: "gpt-4o",
          messages: [
            { role: "system", content: systemPrompt },
            { role: "user", content },
          ],
          max_completion_tokens: 500,
        });

        assistantResponse = completion.choices[0]?.message?.content || 
          "I apologize, but I couldn't generate a response. Please try again.";
      } catch (aiError) {
        console.error("AI error:", aiError);
        assistantResponse = `Based on your query about "${content}", I can help you explore the drilling data. The system currently has ${(await storage.getDashboardStats()).totalFiles} files with various drilling metrics. Would you like me to suggest specific queries you can run in the Query Builder?`;
      }

      const assistantMessage = await storage.createChatMessage({
        userId,
        role: "assistant",
        content: assistantResponse,
      });

      res.json(assistantMessage);
    } catch (error) {
      console.error("Error processing chat:", error);
      res.status(500).json({ message: "Failed to process message" });
    }
  });

  app.delete("/api/chat/clear", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.status(401).json({ message: "Unauthorized" });
      }

      await storage.clearChatMessages(userId);
      res.json({ success: true });
    } catch (error) {
      console.error("Error clearing chat:", error);
      res.status(500).json({ message: "Failed to clear chat" });
    }
  });

  app.get("/api/reports/saved", isAuthenticated, async (req: any, res) => {
    try {
      const userId = req.user?.claims?.sub;
      if (!userId) {
        return res.json([]);
      }

      const queries = await storage.getSavedQueries(userId);
      res.json(queries.map((q) => ({
        id: q.id,
        name: q.name,
        createdAt: q.createdAt?.toISOString().split("T")[0] || "",
      })));
    } catch (error) {
      console.error("Error fetching saved reports:", error);
      res.status(500).json({ message: "Failed to fetch reports" });
    }
  });

  return httpServer;
}
