import { db } from "./db";
import { eq, and, gte, lte, like, desc, sql, count } from "drizzle-orm";
import {
  multipleFiles,
  firstTable,
  secondTable,
  thirdTable,
  fourthTable,
  bitData,
  drillString,
  drillStringFloats,
  mudData,
  mudTreatment,
  companies,
  lostTimeDetails,
  trucks,
  bulks,
  miscellaneous,
  repairTable,
  outOfScope,
  otherDataTable,
  savedQueries,
  chatMessages,
  type MultipleFile,
  type InsertFile,
  type SavedQuery,
  type InsertSavedQuery,
  type ChatMessage,
  type InsertChatMessage,
} from "@shared/schema";

export interface IStorage {
  getFiles(): Promise<MultipleFile[]>;
  getRecentFiles(limit?: number): Promise<MultipleFile[]>;
  getFileById(id: number): Promise<MultipleFile | undefined>;
  createFile(file: InsertFile): Promise<MultipleFile>;
  deleteFile(id: number): Promise<void>;
  updateFileParsed(id: number, isParsed: boolean): Promise<void>;

  getSavedQueries(userId: string): Promise<SavedQuery[]>;
  createSavedQuery(query: InsertSavedQuery): Promise<SavedQuery>;
  deleteSavedQuery(id: number, userId: string): Promise<void>;

  getChatMessages(userId: string): Promise<ChatMessage[]>;
  createChatMessage(message: InsertChatMessage): Promise<ChatMessage>;
  clearChatMessages(userId: string): Promise<void>;

  getDashboardStats(): Promise<{
    totalFiles: number;
    totalRecords: number;
    recentUploads: number;
    avgProcessingTime: string;
  }>;

  getTableStats(): Promise<{ name: string; count: number }[]>;

  getUniqueRigNames(): Promise<string[]>;

  executeQuery(params: {
    tables: Record<string, string[]>;
    rigName?: string;
    dateFrom?: string;
    dateTo?: string;
  }): Promise<{ columns: string[]; rows: Record<string, any>[]; totalRows: number }>;
}

const tableMap: Record<string, any> = {
  first_table: firstTable,
  second_table: secondTable,
  third_table: thirdTable,
  fourth_table: fourthTable,
  bit_data: bitData,
  drill_string: drillString,
  drill_string_floats: drillStringFloats,
  mud_data: mudData,
  mud_treatment: mudTreatment,
  companies: companies,
  lost_time_details: lostTimeDetails,
  trucks: trucks,
  bulks: bulks,
  miscellaneous: miscellaneous,
  repair_table: repairTable,
  out_of_scope: outOfScope,
  other_data_table: otherDataTable,
};

export class DatabaseStorage implements IStorage {
  async getFiles(): Promise<MultipleFile[]> {
    return db.select().from(multipleFiles).orderBy(desc(multipleFiles.uploadDate));
  }

  async getRecentFiles(limit = 10): Promise<MultipleFile[]> {
    return db
      .select()
      .from(multipleFiles)
      .orderBy(desc(multipleFiles.uploadDate))
      .limit(limit);
  }

  async getFileById(id: number): Promise<MultipleFile | undefined> {
    const [file] = await db.select().from(multipleFiles).where(eq(multipleFiles.id, id));
    return file;
  }

  async createFile(file: InsertFile): Promise<MultipleFile> {
    const [newFile] = await db.insert(multipleFiles).values(file).returning();
    return newFile;
  }

  async deleteFile(id: number): Promise<void> {
    await db.delete(multipleFiles).where(eq(multipleFiles.id, id));
  }

  async updateFileParsed(id: number, isParsed: boolean): Promise<void> {
    await db.update(multipleFiles).set({ isParsed }).where(eq(multipleFiles.id, id));
  }

  async getSavedQueries(userId: string): Promise<SavedQuery[]> {
    return db
      .select()
      .from(savedQueries)
      .where(eq(savedQueries.userId, userId))
      .orderBy(desc(savedQueries.createdAt));
  }

  async createSavedQuery(query: InsertSavedQuery): Promise<SavedQuery> {
    const [newQuery] = await db.insert(savedQueries).values(query).returning();
    return newQuery;
  }

  async deleteSavedQuery(id: number, userId: string): Promise<void> {
    await db.delete(savedQueries).where(
      and(eq(savedQueries.id, id), eq(savedQueries.userId, userId))
    );
  }

  async getChatMessages(userId: string): Promise<ChatMessage[]> {
    return db
      .select()
      .from(chatMessages)
      .where(eq(chatMessages.userId, userId))
      .orderBy(chatMessages.createdAt);
  }

  async createChatMessage(message: InsertChatMessage): Promise<ChatMessage> {
    const [newMessage] = await db.insert(chatMessages).values(message).returning();
    return newMessage;
  }

  async clearChatMessages(userId: string): Promise<void> {
    await db.delete(chatMessages).where(eq(chatMessages.userId, userId));
  }

  async getDashboardStats(): Promise<{
    totalFiles: number;
    totalRecords: number;
    recentUploads: number;
    avgProcessingTime: string;
  }> {
    const [fileCount] = await db.select({ count: count() }).from(multipleFiles);
    
    const sevenDaysAgo = new Date();
    sevenDaysAgo.setDate(sevenDaysAgo.getDate() - 7);
    
    const [recentCount] = await db
      .select({ count: count() })
      .from(multipleFiles)
      .where(gte(multipleFiles.createdAt, sevenDaysAgo));

    let totalRecords = 0;
    for (const table of Object.values(tableMap)) {
      try {
        const [result] = await db.select({ count: count() }).from(table);
        totalRecords += result.count;
      } catch {
        // Table might not exist yet
      }
    }

    return {
      totalFiles: fileCount.count,
      totalRecords,
      recentUploads: recentCount.count,
      avgProcessingTime: "2.3s",
    };
  }

  async getTableStats(): Promise<{ name: string; count: number }[]> {
    const stats: { name: string; count: number }[] = [];
    
    const tableNames: Record<string, string> = {
      bit_data: "Bit Data",
      drill_string: "Drill String",
      mud_data: "Mud Data",
      companies: "Companies",
      lost_time_details: "Lost Time",
    };

    for (const [key, displayName] of Object.entries(tableNames)) {
      const table = tableMap[key];
      if (table) {
        try {
          const [result] = await db.select({ count: count() }).from(table);
          stats.push({ name: displayName, count: result.count });
        } catch {
          stats.push({ name: displayName, count: 0 });
        }
      }
    }

    return stats;
  }

  async getUniqueRigNames(): Promise<string[]> {
    const results = await db
      .selectDistinct({ regName: multipleFiles.regName })
      .from(multipleFiles)
      .orderBy(multipleFiles.regName);
    return results
      .map(r => r.regName)
      .filter(name => name && name.trim() !== "" && name !== "Unknown Rig");
  }

  async executeQuery(params: {
    tables: Record<string, string[]>;
    rigName?: string;
    dateFrom?: string;
    dateTo?: string;
  }): Promise<{ columns: string[]; rows: Record<string, any>[]; totalRows: number }> {
    const { tables, rigName, dateFrom, dateTo } = params;
    const allColumns: string[] = [];
    const allRows: Record<string, any>[] = [];

    let filteredFileIds: number[] | null = null;

    if (rigName || dateFrom || dateTo) {
      const fileConditions: any[] = [];

      if (rigName) {
        const rigNames = rigName.split(",").map((r) => r.trim().toLowerCase());
        const likeConditions = rigNames.map((r) => 
          like(sql`LOWER(${multipleFiles.regName})`, `%${r}%`)
        );
        if (likeConditions.length === 1) {
          fileConditions.push(likeConditions[0]);
        } else if (likeConditions.length > 1) {
          fileConditions.push(sql`(${sql.join(likeConditions, sql` OR `)})`);
        }
      }

      if (dateFrom) {
        fileConditions.push(gte(multipleFiles.uploadDate, dateFrom));
      }

      if (dateTo) {
        fileConditions.push(lte(multipleFiles.uploadDate, dateTo));
      }

      const whereClause = fileConditions.length === 1 
        ? fileConditions[0] 
        : and(...fileConditions);

      const matchingFiles = await db
        .select({ id: multipleFiles.id })
        .from(multipleFiles)
        .where(whereClause);

      filteredFileIds = matchingFiles.map((f) => f.id);

      if (filteredFileIds.length === 0) {
        return { columns: [], rows: [], totalRows: 0 };
      }
    }

    for (const [tableName, columns] of Object.entries(tables)) {
      const table = tableMap[tableName];
      if (!table) continue;

      const selectColumns: Record<string, any> = {};
      for (const col of columns) {
        if (table[col]) {
          const displayCol = `${tableName}.${col}`;
          selectColumns[displayCol] = table[col];
          if (!allColumns.includes(displayCol)) {
            allColumns.push(displayCol);
          }
        }
      }

      if (Object.keys(selectColumns).length === 0) continue;

      try {
        let results;

        if (filteredFileIds !== null && table.fileId) {
          const inClause = sql`${table.fileId} IN (${sql.join(
            filteredFileIds.map((id) => sql`${id}`),
            sql`, `
          )})`;
          results = await db
            .select(selectColumns)
            .from(table)
            .where(inClause)
            .limit(1000);
        } else {
          results = await db
            .select(selectColumns)
            .from(table)
            .limit(1000);
        }

        allRows.push(...results);
      } catch (error) {
        console.error(`Error querying ${tableName}:`, error);
      }
    }

    return {
      columns: allColumns,
      rows: allRows,
      totalRows: allRows.length,
    };
  }
}

export const storage = new DatabaseStorage();
