import { sql, relations } from "drizzle-orm";
import { pgTable, text, varchar, integer, timestamp, date, boolean, decimal, serial } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";

// Re-export auth models
export * from "./models/auth";

// Re-export chat models for AI conversations
export * from "./models/chat";

// ============================================
// ARMS - Advanced Reporting Management System
// Drilling Reports Data Schema
// ============================================

// Multiple Files - Main file tracking table
export const multipleFiles = pgTable("multiple_files", {
  id: serial("id").primaryKey(),
  regName: varchar("reg_name", { length: 191 }).notNull(),
  secName: varchar("sec_name", { length: 191 }).notNull(),
  filePath: varchar("file_path", { length: 191 }).notNull(),
  uploadDate: date("upload_date").notNull(),
  isParsed: boolean("is_parsed").default(false),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// First Table - Rig Details
export const firstTable = pgTable("first_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  date: text("date"),
  wellNoAndCharge: text("well_no_and_charge"),
  wellbores: text("wellbores"),
  objective: varchar("objective", { length: 500 }),
  rig: text("rig"),
  foremanEngineer: text("foreman_engineer"),
  engineer: varchar("engineer", { length: 500 }),
  manager: varchar("manager", { length: 500 }),
  thurayaRigForman: text("thuraya_rig_forman"),
  rigFormanVsat: varchar("rig_forman_vsat", { length: 500 }),
  contractorClerk: varchar("contractor_clerk", { length: 500 }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Second Table - Current Operation
export const secondTable = pgTable("second_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  depth: text("depth"),
  last24HrOperations: text("last_24_hr_operations"),
  next24HrPlan: text("next_24_hr_plan"),
  location: text("location"),
  nextLocation: text("next_location"),
  daysSinceSpudAndDate: text("days_since_spud_and_date"),
  formationTops: text("formation_tops"),
  prevDepth: text("prev_depth"),
  depthComments: text("depth_comments"),
  linerSizeTolMdTvd: text("liner_size_tol_md_tvd"),
  lastCsgSizeMdTvd: text("last_csg_size_md_tvd"),
  footage: text("footage"),
  distanceFromDha: text("distance_from_dha"),
  circ: text("circ"),
  presDepth: text("pres_depth"),
  progress: text("progress"),
  casingDepth: text("casing_depth"),
  linerDepth: text("liner_depth"),
  rkhDepth: text("rkh_depth"),
  currentOperation: text("current_operation"),
  nextOperation: text("next_operation"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Third Table - Detailed Operation
export const thirdTable = pgTable("third_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  hrs: text("hrs"),
  fromTo: text("from_to"),
  lateral: text("lateral"),
  holeSize: text("hole_size"),
  cat: text("cat"),
  majorOp: text("major_op"),
  action: text("action"),
  object: text("object"),
  respCo: text("resp_co"),
  holeDepth: text("hole_depth"),
  eventDepth: text("event_depth"),
  summaryOfOperations: text("summary_of_operations"),
  ltType: text("lt_type"),
  ltId: text("lt_id"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Fourth Table - Survey Data / Lateral & TVD
export const fourthTable = pgTable("fourth_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  lateral: text("lateral"),
  surveyMd: text("survey_md"),
  angle: text("angle"),
  azimuth: text("azimuth"),
  cLength: text("c_length"),
  tvd: text("tvd"),
  nsCoordinate: text("ns_coordinate"),
  ewCoordinate: text("ew_coordinate"),
  verticalSec: text("vertical_sec"),
  dls: text("dls"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Bit Data Table
export const bitData = pgTable("bit_data", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  bitNumber: text("bit_number"),
  iadc: text("iadc"),
  size: text("size"),
  manufacturer: text("manufacturer"),
  serialNumber: text("serial_number"),
  type: text("type"),
  jets: text("jets"),
  tfa: text("tfa"),
  mdIn: text("md_in"),
  mdOut: text("md_out"),
  tvdOut: text("tvd_out"),
  feet: text("feet"),
  hours: text("hours"),
  averageRop: text("average_rop"),
  wob: text("wob"),
  rpm: text("rpm"),
  iRow: text("i_row"),
  oRow: text("o_row"),
  dc: text("dc"),
  loc: text("loc"),
  b: text("b"),
  g: text("g"),
  pressure: text("pressure"),
  gpm: text("gpm"),
  jetVel: text("jet_vel"),
  dpAv: text("dp_av"),
  dcAv: text("dc_av"),
  bitHhp: text("bit_hhp"),
  other: text("other"),
  poohReason: text("pooh_reason"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Drill String Table
export const drillString = pgTable("drill_string", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  orderNum: text("order_num"),
  component: text("component"),
  provider: text("provider"),
  nominalSize: text("nominal_size"),
  joints: text("joints"),
  odSize: text("od_size"),
  idSize: text("id_size"),
  length: text("length"),
  topThread: text("top_thread"),
  bottomThread: text("bottom_thread"),
  weight: text("weight"),
  grade: text("grade"),
  class: text("class"),
  serial: text("serial"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Drill String Floats
export const drillStringFloats = pgTable("drill_string_floats", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  float: text("float"),
  bhaHrs: text("bha_hrs"),
  stringWt: text("string_wt"),
  pickUp: text("pick_up"),
  slackOff: text("slack_off"),
  rotTorque: text("rot_torque"),
  jars: text("jars"),
  serialHrs: text("serial_hrs"),
  shock: text("shock"),
  sub: text("sub"),
  mudMotor: text("mud_motor"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Mud Data Table
export const mudData = pgTable("mud_data", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  weightPcf: text("weight_pcf"),
  funnelSec: text("funnel_sec"),
  filtrateWlHthp: text("filtrate_wl_hthp"),
  filtrateWlApi: text("filtrate_wl_api"),
  pv: text("pv"),
  yp: text("yp"),
  electricalStabilityVolts: text("electrical_stability_volts"),
  rpm36: text("rpm_3_6"),
  gelsSecMin: text("gels_sec_min"),
  ph: text("ph"),
  caPpm: text("ca_ppm"),
  clPpm: text("cl_ppm"),
  flTemp: text("fl_temp"),
  cakeHthp: text("cake_hthp"),
  cakeApi: text("cake_api"),
  waterVol: text("water_vol"),
  oilVol: text("oil_vol"),
  solidsVol: text("solids_vol"),
  sandVol: text("sand_vol"),
  lgs: text("lgs"),
  mbt: text("mbt"),
  mudType: text("mud_type"),
  pptSpurt: text("ppt_spurt"),
  pptTotal: text("ppt_total"),
  waterWell: text("water_well"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Mud Treatment Table
export const mudTreatment = pgTable("mud_treatment", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  material: varchar("material", { length: 200 }),
  quantity: varchar("quantity", { length: 200 }),
  unit: varchar("unit", { length: 200 }),
  mudTreatmentData: text("mud_treatment"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Companies Table
export const companies = pgTable("companies", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  company: text("company"),
  category: text("category"),
  noOfPersons: text("no_of_persons"),
  operatingHrs: text("operating_hrs"),
  onLocationHrs: text("on_location_hrs"),
  total: varchar("total", { length: 200 }),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Lost Time Details Table
export const lostTimeDetails = pgTable("lost_time_details", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").references(() => multipleFiles.id),
  fromTo: text("from_to"),
  hrs: text("hrs"),
  cumHrs: text("cum_hrs"),
  ltId: text("lt_id"),
  parentLtId: text("parent_lt_id"),
  type: text("type"),
  cause: text("cause"),
  object: text("object"),
  respCo: text("resp_co"),
  depth: text("depth"),
  ltSummary: text("lt_summary"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Trucks/Boats Table
export const trucks = pgTable("trucks", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  truck: text("truck"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Bulks Table
export const bulks = pgTable("bulks", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  bulk: text("bulk"),
  value: text("value"),
  drillWtr: text("drill_wtr"),
  potWtr: text("pot_wtr"),
  fuelBbls: text("fuel_bbls"),
  bariteSx: text("barite_sx"),
  bentoniteSx: text("bentonite_sx"),
  cementGSx: text("cement_g_sx"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Miscellaneous Table
export const miscellaneous = pgTable("miscellaneous", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  bopTest: text("bop_test"),
  bopDrills: text("bop_drills"),
  wind: text("wind"),
  sea: text("sea"),
  weather: text("weather"),
  dslta: text("dslta"),
  safetyMeeting: text("safety_meeting"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Repair Table
export const repairTable = pgTable("repair_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  instrumentation: text("instrumentation"),
  other: text("other"),
  reporter: text("reporter"),
  computer: text("computer"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Out Of Scope Table
export const outOfScope = pgTable("out_of_scope", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  fromTo: text("from_to"),
  hrs: text("hrs"),
  cumHrs: text("cum_hrs"),
  doosId: text("doos_id"),
  type: text("type"),
  category: text("category"),
  holeSection: text("hole_section"),
  lateral: text("lateral"),
  outSummary: text("out_summary"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Other Data Table (Foreman Remarks/Comments)
export const otherDataTable = pgTable("other_data_table", {
  id: serial("id").primaryKey(),
  fileId: integer("file_id").notNull().references(() => multipleFiles.id),
  comment: text("comment"),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Saved Queries Table
export const savedQueries = pgTable("saved_queries", {
  id: serial("id").primaryKey(),
  userId: varchar("user_id", { length: 255 }).notNull(),
  name: varchar("name", { length: 255 }).notNull(),
  queryConfig: text("query_config").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
  updatedAt: timestamp("updated_at").defaultNow(),
});

// Chat Messages for AI RAG
export const chatMessages = pgTable("chat_messages", {
  id: serial("id").primaryKey(),
  userId: varchar("user_id", { length: 255 }).notNull(),
  role: varchar("role", { length: 50 }).notNull(),
  content: text("content").notNull(),
  createdAt: timestamp("created_at").defaultNow(),
});

// ============================================
// Relations
// ============================================

export const multipleFilesRelations = relations(multipleFiles, ({ many }) => ({
  firstTable: many(firstTable),
  secondTable: many(secondTable),
  thirdTable: many(thirdTable),
  fourthTable: many(fourthTable),
  bitData: many(bitData),
  drillString: many(drillString),
  drillStringFloats: many(drillStringFloats),
  mudData: many(mudData),
  mudTreatment: many(mudTreatment),
  companies: many(companies),
  lostTimeDetails: many(lostTimeDetails),
  trucks: many(trucks),
  bulks: many(bulks),
  miscellaneous: many(miscellaneous),
  repairTable: many(repairTable),
  outOfScope: many(outOfScope),
  otherDataTable: many(otherDataTable),
}));

// ============================================
// Insert Schemas and Types
// ============================================

export const insertFileSchema = createInsertSchema(multipleFiles).omit({ id: true, createdAt: true, updatedAt: true });
export type InsertFile = z.infer<typeof insertFileSchema>;
export type MultipleFile = typeof multipleFiles.$inferSelect;

export const insertSavedQuerySchema = createInsertSchema(savedQueries).omit({ id: true, createdAt: true, updatedAt: true });
export type InsertSavedQuery = z.infer<typeof insertSavedQuerySchema>;
export type SavedQuery = typeof savedQueries.$inferSelect;

export const insertChatMessageSchema = createInsertSchema(chatMessages).omit({ id: true, createdAt: true });
export type InsertChatMessage = z.infer<typeof insertChatMessageSchema>;
export type ChatMessage = typeof chatMessages.$inferSelect;

// Types for all tables
export type FirstTable = typeof firstTable.$inferSelect;
export type SecondTable = typeof secondTable.$inferSelect;
export type ThirdTable = typeof thirdTable.$inferSelect;
export type FourthTable = typeof fourthTable.$inferSelect;
export type BitData = typeof bitData.$inferSelect;
export type DrillString = typeof drillString.$inferSelect;
export type DrillStringFloat = typeof drillStringFloats.$inferSelect;
export type MudData = typeof mudData.$inferSelect;
export type MudTreatment = typeof mudTreatment.$inferSelect;
export type Company = typeof companies.$inferSelect;
export type LostTimeDetail = typeof lostTimeDetails.$inferSelect;
export type Truck = typeof trucks.$inferSelect;
export type Bulk = typeof bulks.$inferSelect;
export type Miscellaneous = typeof miscellaneous.$inferSelect;
export type RepairTable = typeof repairTable.$inferSelect;
export type OutOfScope = typeof outOfScope.$inferSelect;
export type OtherDataTable = typeof otherDataTable.$inferSelect;
